package mezz.jei.gui.ingredients;

import mezz.jei.api.helpers.IModIdHelper;
import mezz.jei.api.ingredients.IIngredientHelper;
import mezz.jei.api.ingredients.IIngredientRenderer;
import mezz.jei.api.ingredients.ITypedIngredient;
import mezz.jei.api.runtime.IIngredientManager;
import mezz.jei.common.config.IIngredientFilterConfig;
import mezz.jei.common.util.SafeIngredientUtil;
import mezz.jei.common.util.StringUtil;
import mezz.jei.common.util.Translator;
import net.minecraft.class_1761;
import net.minecraft.class_1799;
import net.minecraft.class_1836;
import net.minecraft.class_2561;
import net.minecraft.class_2960;
import net.minecraft.class_5348;
import net.minecraft.class_7706;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.jetbrains.annotations.Nullable;
import org.jetbrains.annotations.Unmodifiable;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Set;
import java.util.stream.Stream;

public class ListElementInfo<V> implements IListElementInfo<V> {
	private static final Logger LOGGER = LogManager.getLogger();
	private static int elementCount = 0;

	private final IListElement<V> element;
	private final List<String> names;
	private final List<String> modIds;
	private final List<String> modNames;
	private final class_2960 resourceLocation;

	@Nullable
	public static <V> IListElementInfo<V> create(ITypedIngredient<V> value, IIngredientManager ingredientManager, IModIdHelper modIdHelper) {
		int createdIndex = elementCount++;
		ListElement<V> element = new ListElement<>(value, createdIndex);
		return createFromElement(element, ingredientManager, modIdHelper);
	}

	@Nullable
	public static <V> IListElementInfo<V> createFromElement(IListElement<V> element, IIngredientManager ingredientManager, IModIdHelper modIdHelper) {
		try {
			return new ListElementInfo<>(element, ingredientManager, modIdHelper);
		} catch (RuntimeException e) {
			try {
				ITypedIngredient<V> typedIngredient = element.getTypedIngredient();
				IIngredientHelper<V> ingredientHelper = ingredientManager.getIngredientHelper(typedIngredient.getType());
				String ingredientInfo = ingredientHelper.getErrorInfo(typedIngredient.getIngredient());
				LOGGER.warn("Found a broken ingredient {}", ingredientInfo, e);
			} catch (RuntimeException e2) {
				LOGGER.warn("Found a broken ingredient.", e2);
			}
			return null;
		}
	}

	protected ListElementInfo(IListElement<V> element, IIngredientManager ingredientManager, IModIdHelper modIdHelper) {
		this.element = element;
		ITypedIngredient<V> value = element.getTypedIngredient();
		V ingredient = value.getIngredient();
		IIngredientHelper<V> ingredientHelper = ingredientManager.getIngredientHelper(value.getType());
		this.resourceLocation = ingredientHelper.getResourceLocation(ingredient);
		String displayModId = ingredientHelper.getDisplayModId(ingredient);
		String modId = this.resourceLocation.method_12836();
		if (modId.equals(displayModId)) {
			this.modIds = List.of(modId);
			this.modNames = List.of(modIdHelper.getModNameForModId(modId));
		} else {
			this.modIds = List.of(modId, displayModId);
			this.modNames = List.of(
				modIdHelper.getModNameForModId(modId),
				modIdHelper.getModNameForModId(displayModId)
			);
		}

		String displayNameLowercase = DisplayNameUtil.getLowercaseDisplayNameForSearch(ingredient, ingredientHelper);
		Collection<String> aliases = ingredientManager.getIngredientAliases(value);
		if (aliases.isEmpty()) {
			this.names = List.of(displayNameLowercase);
		} else {
			this.names = new ArrayList<>(1 + aliases.size());
			this.names.add(displayNameLowercase);
			for (String alias : aliases) {
				String lowercaseAlias = Translator.toLowercaseWithLocale(alias);
				this.names.add(lowercaseAlias);
			}
		}
	}

	@Override
	public List<String> getNames() {
		return names;
	}

	@Override
	public String getModNameForSorting() {
		return modNames.getFirst();
	}

	@Override
	public List<String> getModNames() {
		return modNames;
	}

	@Override
	public List<String> getModIds() {
		return modIds;
	}

	@Override
	@Unmodifiable
	public final Set<String> getTooltipStrings(IIngredientFilterConfig config, IIngredientManager ingredientManager) {
		ITypedIngredient<V> value = element.getTypedIngredient();
		IIngredientRenderer<V> ingredientRenderer = ingredientManager.getIngredientRenderer(value.getType());
		class_1836.class_1837 tooltipFlag = config.getSearchAdvancedTooltips() ? class_1836.class_1837.field_41071 : class_1836.class_1837.field_41070;
		tooltipFlag = tooltipFlag.method_47371();

		List<class_2561> tooltip = SafeIngredientUtil.getPlainTooltipForSearch(ingredientManager, ingredientRenderer, value, tooltipFlag);
		Set<String> strings = getStrings(tooltip);

		strings.remove(this.names.getFirst());
		strings.remove(this.modNames.getFirst().toLowerCase(Locale.ENGLISH));
		strings.remove(this.modIds.getFirst());
		strings.remove(resourceLocation.method_12832());

		return strings;
	}

	public static Set<String> getStrings(@Unmodifiable List<class_2561> tooltip) {
		Set<String> result = new HashSet<>();
		for (class_5348 component : tooltip) {
			String string = component.getString();
			string = StringUtil.removeChatFormatting(string);
			string = Translator.toLowercaseWithLocale(string);
			// Split tooltip strings into words to keep them from being too long.
			// Longer strings are more expensive for the suffix tree to handle.
			String[] strings = string.split(" ");
			Collections.addAll(result, strings);
		}
		return result;
	}

	@Override
	public Collection<String> getTagStrings(IIngredientManager ingredientManager) {
		ITypedIngredient<V> value = element.getTypedIngredient();
		IIngredientHelper<V> ingredientHelper = ingredientManager.getIngredientHelper(value.getType());
		return ingredientHelper.getTagStream(value.getIngredient())
			.map(class_2960::method_12832)
			.toList();
	}

	@Override
	public Stream<class_2960> getTagIds(IIngredientManager ingredientManager) {
		ITypedIngredient<V> value = element.getTypedIngredient();
		IIngredientHelper<V> ingredientHelper = ingredientManager.getIngredientHelper(value.getType());
		return ingredientHelper.getTagStream(value.getIngredient());
	}

	@Override
	public Iterable<Integer> getColors(IIngredientManager ingredientManager) {
		ITypedIngredient<V> value = element.getTypedIngredient();
		IIngredientHelper<V> ingredientHelper = ingredientManager.getIngredientHelper(value.getType());
		V ingredient = value.getIngredient();
		return ingredientHelper.getColors(ingredient);
	}

	@Override
	public @Unmodifiable Collection<String> getCreativeTabsStrings(IIngredientManager ingredientManager) {
		class_1799 itemStack = element.getTypedIngredient().getItemStack().orElse(class_1799.field_8037);
		if (itemStack.method_7960()) {
			return List.of();
		}
		Set<String> creativeTabStrings = new HashSet<>();
		for (class_1761 itemGroup : class_7706.method_47341()) {
			if (!itemGroup.method_47311() || itemGroup.method_47312() != class_1761.class_7916.field_41052) {
				continue;
			}
			if (itemGroup.method_45412(itemStack)) {
				String name = itemGroup.method_7737().getString();
				name = StringUtil.removeChatFormatting(name);
				name = Translator.toLowercaseWithLocale(name);
				Collections.addAll(creativeTabStrings, name.split(" "));
			}
		}
		return creativeTabStrings;
	}

	@Override
	public class_2960 getResourceLocation() {
		return resourceLocation;
	}

	@Override
	public IListElement<V> getElement() {
		return element;
	}

	@Override
	public ITypedIngredient<V> getTypedIngredient() {
		return element.getTypedIngredient();
	}

	@Override
	public int getCreatedIndex() {
		return element.getCreatedIndex();
	}
}
