package mezz.jei.common.gui;

import com.google.gson.JsonElement;
import com.mojang.datafixers.util.Either;
import com.mojang.serialization.Codec;
import com.mojang.serialization.DataResult;
import com.mojang.serialization.JsonOps;
import mezz.jei.api.gui.builder.ITooltipBuilder;
import mezz.jei.api.helpers.IJeiHelpers;
import mezz.jei.api.helpers.IModIdHelper;
import mezz.jei.api.ingredients.IIngredientHelper;
import mezz.jei.api.ingredients.IIngredientRenderer;
import mezz.jei.api.ingredients.IIngredientType;
import mezz.jei.api.ingredients.ITypedIngredient;
import mezz.jei.api.ingredients.subtypes.UidContext;
import mezz.jei.api.runtime.IIngredientManager;
import mezz.jei.api.runtime.IJeiKeyMapping;
import mezz.jei.common.Internal;
import mezz.jei.common.config.DebugConfig;
import mezz.jei.common.platform.IPlatformRenderHelper;
import mezz.jei.common.platform.Services;
import mezz.jei.common.util.ErrorUtil;
import net.minecraft.class_124;
import net.minecraft.class_128;
import net.minecraft.class_148;
import net.minecraft.class_1799;
import net.minecraft.class_2561;
import net.minecraft.class_310;
import net.minecraft.class_327;
import net.minecraft.class_332;
import net.minecraft.class_5250;
import net.minecraft.class_5348;
import net.minecraft.class_5455;
import net.minecraft.class_5632;
import net.minecraft.class_638;
import net.minecraft.class_6903;
import org.jetbrains.annotations.Nullable;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.stream.Collectors;

public class JeiTooltip implements ITooltipBuilder {
	private final List<Either<class_5348, class_5632>> lines = new ArrayList<>();
	private @Nullable ITypedIngredient<?> typedIngredient;

	@Override
	public void add(@Nullable class_5348 formattedText) {
		if (formattedText == null) {
			if (Services.PLATFORM.getModHelper().isInDev()) {
				throw new NullPointerException("Tried to add null tooltip text");
			}
			return;
		}
		lines.add(Either.left(formattedText));
	}

	@Override
	public void add(@Nullable class_5632 component) {
		if (component == null) {
			if (Services.PLATFORM.getModHelper().isInDev()) {
				throw new NullPointerException("Tried to add null tooltip component");
			}
			return;
		}
		lines.add(Either.right(component));
	}

	@Override
	public void setIngredient(ITypedIngredient<?> typedIngredient) {
		this.typedIngredient = typedIngredient;
	}

	@Override
	public void addKeyUsageComponent(String translationKey, IJeiKeyMapping keyMapping) {
		class_5250 translatedKeyMessage = keyMapping.getTranslatedKeyMessage().method_27661();
		addKeyUsageComponent(translationKey, translatedKeyMessage);
	}

	public void addKeyUsageComponent(String translationKey, class_5250 keyMapping) {
		class_2561 boldKeyMapping = keyMapping.method_27692(class_124.field_1067);
		class_5250 component = class_2561.method_43469(translationKey, boldKeyMapping)
			.method_27692(class_124.field_1056)
			.method_27692(class_124.field_1080);

		add(component);
	}

	@Override
	public void addAll(Collection<? extends class_5348> components) {
		for (class_5348 component : components) {
			add(component);
		}
	}

	@Override
	public void clearIngredient() {
		this.typedIngredient = null;
	}

	@Override
	public List<Either<class_5348, class_5632>> getLines() {
		return lines;
	}

	public void addAll(JeiTooltip tooltip) {
		lines.addAll(tooltip.lines);
	}

	public boolean isEmpty() {
		return lines.isEmpty() && typedIngredient == null;
	}

	@Deprecated
	public List<class_2561> getLegacyComponents() {
		return lines.stream()
			.<class_2561>mapMulti((e, consumer) -> {
				e.left().ifPresent(f -> {
					if (f instanceof class_2561 c) {
						consumer.accept(c);
					}
				});
			})
			.collect(Collectors.toCollection(ArrayList::new));
	}

	@Override
	public String toString() {
		return lines.stream()
			.map(e -> e.map(
				class_5348::getString,
				Object::toString
			))
			.collect(Collectors.joining("\n", "[\n", "\n]"));
	}

	public void draw(class_332 guiGraphics, int x, int y) {
		if (typedIngredient != null) {
			draw(guiGraphics, x, y, typedIngredient);
			return;
		}
		if (isEmpty()) {
			return;
		}
		class_310 minecraft = class_310.method_1551();
		class_327 font = minecraft.field_1772;
		IPlatformRenderHelper renderHelper = Services.PLATFORM.getRenderHelper();
		try {
			renderHelper.renderTooltip(guiGraphics, lines, x, y, font, class_1799.field_8037);
		} catch (RuntimeException e) {
			throw new RuntimeException("Crashed when rendering tooltip:\n" + this, e);
		}
	}

	private <T> void draw(class_332 guiGraphics, int x, int y, ITypedIngredient<T> typedIngredient) {
		IIngredientType<T> ingredientType = typedIngredient.getType();
		IIngredientManager ingredientManager = Internal.getJeiRuntime().getIngredientManager();
		IIngredientRenderer<T> ingredientRenderer = ingredientManager.getIngredientRenderer(ingredientType);
		draw(guiGraphics, x, y, typedIngredient, ingredientRenderer, ingredientManager);
	}

	public <T> void draw(
		class_332 guiGraphics,
		int x,
		int y,
		ITypedIngredient<T> typedIngredient,
		IIngredientRenderer<T> ingredientRenderer,
		IIngredientManager ingredientManager
	) {
		class_310 minecraft = class_310.method_1551();
		T ingredient = typedIngredient.getIngredient();
		class_327 font = ingredientRenderer.getFontRenderer(minecraft, ingredient);
		class_1799 itemStack = typedIngredient.getItemStack().orElse(class_1799.field_8037);

		itemStack.method_32347()
			.ifPresent((c) -> {
				lines.add(1, Either.right(c));
			});

		addDebugInfo(ingredientManager, typedIngredient);

		IJeiHelpers jeiHelpers = Internal.getJeiRuntime().getJeiHelpers();
		IModIdHelper modIdHelper = jeiHelpers.getModIdHelper();
		modIdHelper.getModNameForTooltip(typedIngredient)
			.ifPresent(this::add);

		if (isEmpty()) {
			return;
		}
		try {
			IPlatformRenderHelper renderHelper = Services.PLATFORM.getRenderHelper();
			renderHelper.renderTooltip(guiGraphics, lines, x, y, font, itemStack);
		} catch (RuntimeException e) {
			class_128 crashReport = ErrorUtil.createIngredientCrashReport(e, "Rendering ingredient tooltip", ingredientManager, typedIngredient);
			crashReport.method_562("tooltip")
				.method_578("value", this);
			throw new class_148(crashReport);
		}
	}

	private <T> void addDebugInfo(IIngredientManager ingredientManager,  ITypedIngredient<T> typedIngredient) {
		if (!DebugConfig.isDebugInfoTooltipsEnabled() || !class_310.method_1551().field_1690.field_1827) {
			return;
		}
		T ingredient = typedIngredient.getIngredient();
		IIngredientType<T> type = typedIngredient.getType();
		IIngredientHelper<T> ingredientHelper = ingredientManager.getIngredientHelper(type);
		Codec<T> ingredientCodec = ingredientManager.getIngredientCodec(type);

		add(class_2561.method_43473());
		add(
			class_2561.method_43470("JEI Debug:")
				.method_27692(class_124.field_1063)
		);
		add(
			class_2561.method_43470("• type: " + ingredientHelper.getIngredientType().getUid())
				.method_27692(class_124.field_1063)
		);
		add(
			class_2561.method_43470("• has subtypes: " + (ingredientHelper.hasSubtypes(ingredient) ? "true" : "false"))
				.method_27692(class_124.field_1063)
		);
		add(
			class_2561.method_43470("• uid: " + ingredientHelper.getUid(ingredient, UidContext.Ingredient))
				.method_27692(class_124.field_1063)
		);
		try {
			class_310 minecraft = class_310.method_1551();
			class_638 level = minecraft.field_1687;
			assert level != null;
			class_5455 registryAccess = level.method_30349();
			class_6903<JsonElement> registryOps = registryAccess.method_57093(JsonOps.INSTANCE);
			String jsonResult = ingredientCodec.encodeStart(registryOps, ingredient)
					.mapOrElse(
						JsonElement::toString,
						DataResult.Error::message
					);
			add(
				class_2561.method_43470("• json: " + jsonResult)
					.method_27692(class_124.field_1063)
			);
		} catch (RuntimeException e) {
			add(
				class_2561.method_43470("• json crashed: " + e.getMessage())
					.method_27692(class_124.field_1079)
			);
		}
		add(
			class_2561.method_43470("• extra info: " + ingredientHelper.getErrorInfo(ingredient))
				.method_27692(class_124.field_1063)
		);
		add(class_2561.method_43473());
	}
}
