package mezz.jei.library.helpers;

import com.mojang.datafixers.util.Either;
import com.mojang.datafixers.util.Pair;
import com.mojang.serialization.Codec;
import com.mojang.serialization.DataResult;
import com.mojang.serialization.MapCodec;
import com.mojang.serialization.codecs.RecordCodecBuilder;
import mezz.jei.api.helpers.ICodecHelper;
import mezz.jei.api.ingredients.IIngredientSupplier;
import mezz.jei.api.ingredients.IIngredientType;
import mezz.jei.api.ingredients.ITypedIngredient;
import mezz.jei.api.recipe.IFocus;
import mezz.jei.api.recipe.IFocusFactory;
import mezz.jei.api.recipe.IRecipeManager;
import mezz.jei.api.recipe.RecipeIngredientRole;
import mezz.jei.api.recipe.RecipeType;
import mezz.jei.api.recipe.category.IRecipeCategory;
import mezz.jei.api.runtime.IIngredientManager;
import mezz.jei.common.codecs.TupleCodec;
import mezz.jei.common.codecs.TypedIngredientCodecs;
import net.minecraft.class_1860;
import net.minecraft.class_1863;
import net.minecraft.class_2960;
import net.minecraft.class_310;
import net.minecraft.class_638;
import net.minecraft.class_8786;
import org.jetbrains.annotations.Nullable;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

public class CodecHelper implements ICodecHelper {
	private static final Codec<class_8786<?>> RECIPE_HOLDER_CODEC = Codec.lazyInitialized(() -> {
		class_310 minecraft = class_310.method_1551();
		class_638 level = minecraft.field_1687;
		assert level != null;
		class_1863 recipeManager = level.method_8433();

		return Codec.either(
			class_2960.field_25139,
			TupleCodec.of(
				class_2960.field_25139,
				class_1860.field_47319
			)
		)
		.flatXmap(
			either -> {
				return either.map(
					recipeHolderId -> {
						return recipeManager.method_8130(recipeHolderId)
							.map(DataResult::success)
							.orElseGet(() -> DataResult.error(() -> "Could not find recipe for key: " + recipeHolderId));
					},
					pair -> {
						class_2960 recipeHolderId = pair.getFirst();
						class_1860<?> recipe = pair.getSecond();
						if (recipe == null) {
							return DataResult.error(() -> "Could not find recipe for key: " + recipeHolderId);
						}
						class_8786<?> recipeHolder = new class_8786<>(recipeHolderId, recipe);
						return DataResult.success(recipeHolder);
					}
				);
			},
			recipeHolder -> {
				class_2960 recipeHolderId = recipeHolder.comp_1932();
				Optional<class_8786<?>> found = recipeManager.method_8130(recipeHolderId);
				if (found.isPresent() && found.get().equals(recipeHolder)) {
					return DataResult.success(Either.left(recipeHolderId));
				}
				class_1860<?> recipe = recipeHolder.comp_1933();
				return DataResult.success(Either.right(Pair.of(recipeHolderId, recipe)));
			}
		);
	});

	private final IIngredientManager ingredientManager;
	private final IFocusFactory focusFactory;
	private final Map<RecipeType<?>, Codec<?>> defaultRecipeCodecs = new HashMap<>();
	private @Nullable Codec<RecipeType<?>> recipeTypeCodec;

	public CodecHelper(IIngredientManager ingredientManager, IFocusFactory focusFactory) {
		this.ingredientManager = ingredientManager;
		this.focusFactory = focusFactory;
	}

	@Override
	public Codec<IIngredientType<?>> getIngredientTypeCodec() {
		return TypedIngredientCodecs.getIngredientTypeCodec(ingredientManager);
	}

	@Override
	public MapCodec<ITypedIngredient<?>> getTypedIngredientCodec() {
		return TypedIngredientCodecs.getIngredientCodec(ingredientManager);
	}

	@Override
	public <T> Codec<ITypedIngredient<T>> getTypedIngredientCodec(IIngredientType<T> ingredientType) {
		return TypedIngredientCodecs.getIngredientCodec(ingredientType, ingredientManager);
	}

	@Override
	public <T extends class_8786<?>> Codec<T> getRecipeHolderCodec() {
		@SuppressWarnings("unchecked")
		Codec<T> recipeHolderCodec = (Codec<T>) RECIPE_HOLDER_CODEC;
		return recipeHolderCodec;
	}

	@Override
	public <T> Codec<T> getSlowRecipeCategoryCodec(IRecipeCategory<T> recipeCategory, IRecipeManager recipeManager) {
		RecipeType<T> recipeType = recipeCategory.getRecipeType();
		@SuppressWarnings("unchecked")
		Codec<T> codec = (Codec<T>) defaultRecipeCodecs.get(recipeType);
		if (codec == null) {
			codec = createDefaultRecipeCategoryCodec(recipeManager, recipeCategory);
			defaultRecipeCodecs.put(recipeType, codec);
		}
		return codec;
	}

	private <T> Codec<T> createDefaultRecipeCategoryCodec(IRecipeManager recipeManager, IRecipeCategory<T> recipeCategory) {
		Codec<Pair<class_2960, ITypedIngredient<?>>> legacyPairCodec = RecordCodecBuilder.create((builder) -> {
			return builder.group(
				class_2960.field_25139.fieldOf("resourceLocation")
					.forGetter(Pair::getFirst),
				getTypedIngredientCodec().codec().fieldOf("output")
					.forGetter(Pair::getSecond)
			).apply(builder, Pair::new);
		});

		Codec<Pair<class_2960, ITypedIngredient<?>>> tupleCodec = TupleCodec.of(
			class_2960.field_25139,
			getTypedIngredientCodec().codec()
		);

		return Codec.withAlternative(tupleCodec, legacyPairCodec)
		.flatXmap(
			pair -> {
				class_2960 registryName = pair.getFirst();
				ITypedIngredient<?> output = pair.getSecond();
				IFocus<?> focus = focusFactory.createFocus(RecipeIngredientRole.OUTPUT, output);

				RecipeType<T> recipeType = recipeCategory.getRecipeType();

				return recipeManager.createRecipeLookup(recipeType)
					.limitFocus(List.of(focus))
					.get()
					.filter(recipe -> registryName.equals(recipeCategory.getRegistryName(recipe)))
					.findFirst()
					.map(DataResult::success)
					.orElseGet(() -> DataResult.error(() -> "No recipe found for registry name: " + registryName));
			},
			recipe -> {
				class_2960 registryName = recipeCategory.getRegistryName(recipe);
				if (registryName == null) {
					return DataResult.error(() -> "No registry name for recipe");
				}
				IIngredientSupplier ingredients = recipeManager.getRecipeIngredients(recipeCategory, recipe);
				List<ITypedIngredient<?>> outputs = ingredients.getIngredients(RecipeIngredientRole.OUTPUT);
				if (outputs.isEmpty()) {
					return DataResult.error(() -> "No outputs for recipe");
				}
				Pair<class_2960, ITypedIngredient<?>> result = new Pair<>(registryName, outputs.getFirst());
				return DataResult.success(result);
			}
		);
	}

	@Override
	public Codec<RecipeType<?>> getRecipeTypeCodec(IRecipeManager recipeManager) {
		if (recipeTypeCodec == null) {
			recipeTypeCodec = class_2960.field_25139.flatXmap(
				resourceLocation -> {
					return recipeManager.getRecipeType(resourceLocation)
						.map(DataResult::success)
						.orElseGet(() -> DataResult.error(() -> "Failed to find recipe type " + resourceLocation));
				},
				recipeType -> {
					class_2960 uid = recipeType.getUid();
					return DataResult.success(uid);
				}
			);
		}
		return recipeTypeCodec;
	}
}
