package com.faux.customentitydata.api.playersaves;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import net.minecraft.class_156;
import net.minecraft.class_1657;
import net.minecraft.class_2487;
import net.minecraft.class_2505;
import net.minecraft.class_2507;
import net.minecraft.class_2960;

/**
 * This class provides a custom player data solution that stores player data in a separate auxiliary file in the player
 * directory. While this is not the definitive way to store custom player data it is often the cleanest approach.
 * <p>
 * Some benefits of this approach include:
 * <ul>
 *     <li>No risk of corrupting vanilla player data.</li>
 *     <li>Custom data survives player respawn.</li>
 *     <li>Custom data survives mod being uninstalled temporarily.</li>
 * </ul>
 * <p>
 * This implementation will invoke {@link #savePlayer(class_1657)} when the vanilla player data is written to disk. You can
 * use this method to pull data from your own in-memory sources and store it in the resulting NBT tag. Once this tag is
 * generated it will be saved to the disk as a compressed NBT file. A backup save file will also be generated and
 * maintained to help players rollback their data in the event of corruption or other adverse circumstances. Loading
 * your custom data is done in {@link #loadPlayer(class_1657, class_2487)} which is invoked after vanilla has loaded their
 * player data. You can use this method to restore your in-memory player data.
 * <p>
 * Files will be saved within subdirectories of the vanilla player folder. These subdirectories are based on the
 * {@link #handlerId} provided when constructing your implementation.
 */
public abstract class CustomPlayerSave implements IPlayerSaveListener, IPlayerLoadListener {

    /**
     * An identifier used when reading and writing custom player data. The ID is incorporated into the save filepath to
     * prevent reduce the likelihood of conflicting entries.
     */
    private final class_2960 handlerId;

    /**
     * A logger instance used to track warnings and debug information about the save data.
     */
    private final Logger log;

    public CustomPlayerSave(class_2960 handlerId) {

        this.handlerId = handlerId;
        this.log = LoggerFactory.getLogger(handlerId.toString());

        IPlayerLoadListener.EVENT.register(this);
        IPlayerSaveListener.EVENT.register(this);
    }

    /**
     * Saves custom data for a player as an NBT. This is invoked after the vanilla player data has been saved.
     *
     * @param player The player being saved.
     * @return The data to save for the player.
     */
    public abstract class_2487 savePlayer(class_1657 player);

    /**
     * Loads custom data for a player from NBT. This is invoked after the vanilla player data has been loaded.
     *
     * @param player   The player being loaded.
     * @param saveData The data for the player. If the player has no existing data an empty tag will be provided.
     */
    public abstract void loadPlayer(class_1657 player, class_2487 saveData);

    @Override
    public void loadPlayerData(class_1657 player, Path saveDir) {

        final long startTime = System.nanoTime();
        final Path customSaveDir = getCustomSaveDir(saveDir);
        class_2487 data = new class_2487();

        // Attempt to read the save file as NBT
        try {

            final Path targetSave = customSaveDir.resolve(player.method_5845() + ".dat");
            if (Files.exists(targetSave) && Files.isRegularFile(targetSave)) {
                data = class_2507.method_30613(targetSave, class_2505.method_53898());
            }
        } catch (IOException e) {

            this.log.error("Failed to read custom data file for player {} ({}).", player.method_5477().getString(), player.method_5845(), e);
        }

        // Attempt to deserialize the NBT data
        try {

            this.loadPlayer(player, data);
            final long endTime = System.nanoTime();
            this.log.debug("Loaded data for {}. Took {}ns.", player.method_5477().getString(), endTime - startTime);
        } catch (Exception e) {

            this.log.error("Failed to read custom data for player {} ({}).", player.method_5477().getString(), player.method_5845(), e);
        }
    }

    @Override
    public void savePlayerData(class_1657 player, Path saveDir) {

        try {

            final long startTime = System.nanoTime();
            final Path customSaveDir = getCustomSaveDir(saveDir);

            // Write save data to a temporary location.
            final Path tempSave = Files.createTempFile(customSaveDir, player.method_5845() + "-", ".dat");
            class_2507.method_30614(this.savePlayer(player), tempSave);

            // Backup existing save and overwrite with new data.
            final Path targetSave = customSaveDir.resolve(player.method_5845() + ".dat");
            final Path backupSave = customSaveDir.resolve(player.method_5845() + ".dat_old");
            class_156.method_30626(targetSave, tempSave, backupSave);
            final long endTime = System.nanoTime();

            this.log.debug("Saved data for {}. Took {}ns.", player.method_5477().getString(), endTime - startTime);
        } catch (IOException e) {

            this.log.error("Failed to write custom data for player {} ({}).", player.method_5477().getString(), player.method_5845(), e);
        }
    }

    /**
     * Generates a subdirectory for custom player files using {@link this#handlerId}. The directory will take on the
     * structure of /{namespace}/{path}/. If the generated directory does not exist it will try to create it.
     *
     * @param saveDir The root save directory. This is intended to be the player save directory.
     * @return The subdirectory for custom player files.
     */
    private Path getCustomSaveDir(Path saveDir) {
        final Path customSaveDir = saveDir.resolve(handlerId.method_12836()).resolve(handlerId.method_12832());

        if (Files.notExists(customSaveDir)) {
            try {
                return Files.createDirectories(customSaveDir);
            } catch (IOException e) {
                this.log.error("Failed to create custom save directory {}.", customSaveDir.toAbsolutePath());
                throw new RuntimeException(e);
            }
        }

        return customSaveDir;
    }
}