package com.blamejared.ironsmelters.registry;

import com.blamejared.ironsmelters.Util;
import com.blamejared.ironsmelters.platform.Services;
import java.util.Collection;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.function.UnaryOperator;
import net.minecraft.class_1747;
import net.minecraft.class_1792;
import net.minecraft.class_2248;
import net.minecraft.class_2378;
import net.minecraft.class_2960;
import net.minecraft.class_4970;
import net.minecraft.class_5321;

/**
 * Utility class for multiloader registration.
 * <p>
 * Example usage:
 * <pre>{@code
 * public static final RegistrationProvider<Test> PROVIDER = RegistrationProvider.get(Test.REGISTRY, "modid");
 * public static final RegistryObject<Test> OBJECT = PROVIDER.register("object", () -> new Test());
 *
 * // The purpose of this method is to be called in the mod's constructor, in order to assure that the class is loaded, and that objects can be registered.
 * public static void loadClass(){}
 * }</pre>
 *
 * @param <T> the type of the objects that this class registers
 */
public interface RegistrationProvider<T> {
    
    /**
     * Gets a provider for specified {@code modId} and {@code resourceKey}. <br>
     * It is <i>recommended</i> to store the resulted provider in a {@code static final} field to
     * the {@link Factory#INSTANCE factory} creating multiple providers for the same resource key and mod id.
     *
     * @param resourceKey the {@link class_5321} of the registry of the provider
     * @param modId       the mod id that the provider will register objects for
     * @param <T>         the type of the provider
     *
     * @return the provider
     */
    static <T> RegistrationProvider<T> get(class_5321<? extends class_2378<T>> resourceKey, String modId) {
        
        return Factory.INSTANCE.create(resourceKey, modId);
    }
    
    /**
     * Gets a provider for specified {@code modId} and {@code registry}. <br>
     * It is <i>recommended</i> to store the resulted provider in a {@code static final} field to
     * the {@link Factory#INSTANCE factory} creating multiple providers for the same resource key and mod id.
     *
     * @param registry the {@link class_2378} of the provider
     * @param modId    the mod id that the provider will register objects for
     * @param <T>      the type of the provider
     *
     * @return the provider
     */
    static <T> RegistrationProvider<T> get(class_2378<T> registry, String modId) {
        
        return Factory.INSTANCE.create(registry, modId);
    }
    
    /**
     * Registers an object.
     *
     * @param name     the name of the object
     * @param supplier a supplier of the object to register
     * @param <I>      the type of the object
     *
     * @return a wrapper containing the lazy registered object. <strong>Calling {@link RegistryObject#get() get} too early
     * on the wrapper might result in crashes!</strong>
     */
    <I extends T> RegistryObject<I> register(String name, Supplier<? extends I> supplier);
    
    default <I extends T> RegistryObject<I> block(String name, Function<class_2248.Properties, ? extends I> supplier) {
        
        return register(name, () -> {
            class_2960 location = class_2960.method_60655(getModId(), name);
            class_5321<class_2248> rk = class_5321.method_29179(Util.uncheck(resourceKey()), location);
            return supplier.apply(class_2248.Properties.method_9637());
        });
    }
    
    default <I extends T> RegistryObject<I> block(String name, Function<class_2248.Properties, ? extends I> supplier, class_4970.class_2251 defaultProperties) {
        
        return register(name, () -> {
            class_2960 location = class_2960.method_60655(getModId(), name);
            class_5321<class_2248> rk = class_5321.method_29179(Util.uncheck(resourceKey()), location);
            return supplier.apply(defaultProperties);
        });
    }
    
    default <I extends T> RegistryObject<I> item(String name, Function<class_1792.class_1793, ? extends I> supplier) {
        
        return register(name, () -> {
            class_2960 location = class_2960.method_60655(getModId(), name);
            class_5321<class_1792> rk = class_5321.method_29179(Util.uncheck(resourceKey()), location);
            return supplier.apply(new class_1792.class_1793());
        });
    }
    
    default <I extends T> RegistryObject<I> item(String name, Function<class_1792.class_1793, ? extends I> supplier, class_1792.class_1793 defaultProperties) {
        
        return register(name, () -> {
            class_2960 location = class_2960.method_60655(getModId(), name);
            class_5321<class_1792> rk = class_5321.method_29179(Util.uncheck(resourceKey()), location);
            return supplier.apply(defaultProperties);
        });
    }
    
    default <I extends T> RegistryObject<I> blockItem(String name, RegistryObject<class_2248> block, UnaryOperator<class_1792.class_1793> supplier) {
        
        return register(name, () -> {
            class_2960 location = class_2960.method_60655(getModId(), name);
            class_5321<class_1792> rk = class_5321.method_29179(Util.uncheck(resourceKey()), location);
            return Util.uncheck(new class_1747(block.get(), supplier.apply(new class_1792.class_1793())));
        });
    }
    
    default <I extends T> RegistryObject<I> blockItem(String name, RegistryObject<class_2248> block, UnaryOperator<class_1792.class_1793> supplier, class_1792.class_1793 defaultProperties) {
        
        return register(name, () -> {
            class_2960 location = class_2960.method_60655(getModId(), name);
            class_5321<class_1792> rk = class_5321.method_29179(Util.uncheck(resourceKey()), location);
            return Util.uncheck(new class_1747(block.get(), supplier.apply(defaultProperties)));
        });
    }
    
    /**
     * Gets all the objects currently registered.
     *
     * @return an <strong>immutable</strong> view of all the objects currently registered
     */
    Collection<RegistryObject<T>> getEntries();
    
    /**
     * Gets the mod id that this provider registers objects for.
     *
     * @return the mod id
     */
    String getModId();
    
    class_5321<? extends class_2378<T>> resourceKey();
    
    /**
     * Factory class for {@link RegistrationProvider registration providers}. <br>
     * This class is loaded using {@link java.util.ServiceLoader Service Loaders}, and only one
     * should exist per mod loader.
     */
    interface Factory {
        
        /**
         * The singleton instance of the {@link Factory}. This is different on each loader.
         */
        Factory INSTANCE = Services.load(Factory.class);
        
        /**
         * Creates a {@link RegistrationProvider}.
         *
         * @param resourceKey the {@link class_5321} of the registry to create this provider for
         * @param modId       the mod id for which the provider will register objects
         * @param <T>         the type of the provider
         *
         * @return the provider
         */
        <T> RegistrationProvider<T> create(class_5321<? extends class_2378<T>> resourceKey, String modId);
        
        /**
         * Creates a {@link RegistrationProvider}.
         *
         * @param registry the {@link class_2378} to create this provider for
         * @param modId    the mod id for which the provider will register objects
         * @param <T>      the type of the provider
         *
         * @return the provider
         */
        default <T> RegistrationProvider<T> create(class_2378<T> registry, String modId) {
            
            return create(registry.method_30517(), modId);
        }
        
    }
    
}
