package com.almostreliable.unified.recipe;

import com.almostreliable.unified.BuildConfig;
import com.almostreliable.unified.utils.Utils;
import com.google.gson.JsonArray;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import net.minecraft.core.RegistryAccess;
import net.minecraft.network.FriendlyByteBuf;
import net.minecraft.resources.ResourceLocation;
import net.minecraft.world.Container;
import net.minecraft.world.item.ItemStack;
import net.minecraft.world.item.crafting.Recipe;
import net.minecraft.world.item.crafting.RecipeSerializer;
import net.minecraft.world.item.crafting.RecipeType;
import net.minecraft.world.level.Level;

import javax.annotation.Nullable;
import java.util.HashMap;
import java.util.Map;

/**
 * This recipe is used to track which recipes were unified. It is NOT used for crafting.
 * Each tracker will hold one namespace with a list of recipes that were unified for it.
 */
public class ClientRecipeTracker implements Recipe<Container> {
    public static final ResourceLocation ID = Utils.getRL("client_recipe_tracker");
    public static final String RECIPES = "recipes";
    public static final String NAMESPACE = "namespace";
    public static final int UNIFIED_FLAG = 1;
    public static final int DUPLICATE_FLAG = 2;
    public static final RecipeSerializer<ClientRecipeTracker> SERIALIZER = new Serializer();
    public static final RecipeType<ClientRecipeTracker> TYPE = new RecipeType<>() {
        @Override
        public String toString() {
            return ID.m_135815_();
        }
    };

    private final ResourceLocation id;
    private final Map<ResourceLocation, ClientRecipeLink> recipes = new HashMap<>();
    private final String namespace;

    protected ClientRecipeTracker(ResourceLocation id, String namespace) {
        this.id = id;
        this.namespace = namespace;
    }

    /**
     * Creates a raw string representation.
     *
     * @param isUnified   Whether the recipe was unified.
     * @param isDuplicate Whether the recipe had duplicates.
     * @param idPath      The path of the recipe.
     * @return String representation as: `flag$idPath`
     */
    private static String createRaw(boolean isUnified, boolean isDuplicate, String idPath) {
        int flag = 0;
        if (isUnified) flag |= UNIFIED_FLAG;
        if (isDuplicate) flag |= DUPLICATE_FLAG;
        return flag + "$" + idPath;
    }

    //<editor-fold defaultstate="collapsed" desc="Default recipe stuff. Ignore this. Forget this.">
    @Override
    public boolean m_5818_(Container container, Level level) {
        return false;
    }

    @Override
    public ItemStack m_5874_(Container container, RegistryAccess registryAccess) {
        return ItemStack.f_41583_;
    }

    @Override
    public boolean m_8004_(int width, int height) {
        return false;
    }

    @Override
    public ItemStack m_8043_(RegistryAccess registryAccess) {
        return ItemStack.f_41583_;
    }

    @Override
    public ResourceLocation m_6423_() {
        return id;
    }
    //</editor-fold>

    @Override
    public RecipeSerializer<?> m_7707_() {
        return SERIALIZER;
    }

    @Override
    public RecipeType<?> m_6671_() {
        return TYPE;
    }

    private void add(ClientRecipeLink clientRecipeLink) {
        recipes.put(clientRecipeLink.id(), clientRecipeLink);
    }

    @Nullable
    public ClientRecipeLink getLink(ResourceLocation recipeId) {
        return recipes.get(recipeId);
    }

    public record ClientRecipeLink(ResourceLocation id, boolean isUnified, boolean isDuplicate) {}

    public static class Serializer implements RecipeSerializer<ClientRecipeTracker> {

        /**
         * Reads a recipe from a json file. Recipe will look like this:
         * <pre>
         * {@code
         * {
         *      "type": "almostunified:client_recipe_tracker",
         *      "namespace": "minecraft", // The namespace of the recipes.
         *      "recipes": [
         *          "flag$recipePath",
         *          "flag$recipe2Path",
         *          ...
         *          "flag$recipeNPath"
         *      ]
         * }
         * }
         * </pre>
         *
         * @param recipeId The id of the recipe for the tracker.
         * @param json     The json object.
         * @return The recipe tracker.
         */
        @Override
        public ClientRecipeTracker m_6729_(ResourceLocation recipeId, JsonObject json) {
            String namespace = json.get(NAMESPACE).getAsString();
            JsonArray recipes = json.get(RECIPES).getAsJsonArray();
            ClientRecipeTracker tracker = new ClientRecipeTracker(recipeId, namespace);
            for (JsonElement element : recipes) {
                ClientRecipeLink clientRecipeLink = parseRaw(namespace, element.getAsString());
                tracker.add(clientRecipeLink);
            }
            return tracker;
        }

        @Override
        public ClientRecipeTracker m_8005_(ResourceLocation recipeId, FriendlyByteBuf buffer) {
            int size = buffer.readInt();
            String namespace = buffer.m_130277_();

            ClientRecipeTracker recipe = new ClientRecipeTracker(recipeId, namespace);
            for (int i = 0; i < size; i++) {
                String raw = buffer.m_130277_();
                ClientRecipeLink clientRecipeLink = parseRaw(namespace, raw);
                recipe.add(clientRecipeLink);
            }
            return recipe;
        }

        /**
         * Writes the tracker to the buffer. The namespace is written separately to save some bytes.
         * Buffer output will look like:
         * <pre>
         *     size
         *     namespace
         *     flag$recipePath
         *     flag$recipe2Path
         *     ...
         *     flag$recipeNPath
         * </pre>
         *
         * @param buffer The buffer to write to
         * @param recipe The recipe to write
         */
        @Override
        public void toNetwork(FriendlyByteBuf buffer, ClientRecipeTracker recipe) {
            buffer.writeInt(recipe.recipes.size());
            buffer.m_130070_(recipe.namespace);
            for (ClientRecipeLink clientRecipeLink : recipe.recipes.values()) {
                String raw = createRaw(clientRecipeLink.isUnified(),
                        clientRecipeLink.isDuplicate(),
                        clientRecipeLink.id().m_135815_());
                buffer.m_130070_(raw);
            }
        }

        /**
         * Creates a {@link ClientRecipeLink} from a raw string for the given namespace.
         *
         * @param namespace The namespace to use.
         * @param raw       The raw string.
         * @return The client sided recipe link.
         */
        private static ClientRecipeLink parseRaw(String namespace, String raw) {
            String[] split = raw.split("\\$", 2);
            int flag = Integer.parseInt(split[0]);
            boolean isUnified = (flag & UNIFIED_FLAG) != 0;
            boolean isDuplicate = (flag & DUPLICATE_FLAG) != 0;
            return new ClientRecipeLink(new ResourceLocation(namespace, split[1]), isUnified, isDuplicate);
        }
    }

    public static class RawBuilder {

        private final Map<String, JsonArray> recipesByNamespace = new HashMap<>();

        public void add(RecipeLink recipe) {
            ResourceLocation recipeId = recipe.getId();
            JsonArray array = recipesByNamespace.computeIfAbsent(recipeId.m_135827_(), k -> new JsonArray());
            array.add(createRaw(recipe.isUnified(), recipe.hasDuplicateLink(), recipeId.m_135815_()));
        }

        /**
         * Creates a map with the namespace as key and the json recipe.
         * These recipes are used later in {@link Serializer#m_6729_(ResourceLocation, JsonObject)}
         *
         * @return The map with the namespace as key and the json recipe.
         */
        public Map<ResourceLocation, JsonObject> compute() {
            Map<ResourceLocation, JsonObject> result = new HashMap<>();
            recipesByNamespace.forEach((namespace, recipes) -> {
                JsonObject json = new JsonObject();
                json.addProperty("type", ID.toString());
                json.addProperty(NAMESPACE, namespace);
                json.add(RECIPES, recipes);
                result.put(new ResourceLocation(BuildConfig.MOD_ID, namespace), json);
            });
            return result;
        }
    }
}
