package com.almostreliable.unified.utils;

import com.almostreliable.unified.api.unification.UnificationEntry;

import org.jetbrains.annotations.Nullable;

import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import net.minecraft.class_2378;
import net.minecraft.class_2960;
import net.minecraft.class_5321;
import net.minecraft.class_6862;
import net.minecraft.class_6880;

/**
 * Base wrapper to store vanilla tags and their holders. The wrapper allows to add new tags and holders to a tag.
 * By default, the holder collection for each tag is immutable. When attempting to modify the collection it will be copied and marked as modified.
 * <p>
 * After all operations are done, the vanilla tags should be sealed with {@link VanillaTagWrapper#seal()} to prevent further changes.
 *
 * @param <T>
 */
public class VanillaTagWrapper<T> {

    private final class_2378<T> registry;
    private final Map<class_2960, Collection<class_6880<T>>> vanillaTags;
    @Nullable
    private Map<class_6880<T>, Set<class_2960>> holdersToTags;
    private final Set<class_2960> modifiedTags = new HashSet<>();

    public static <T> VanillaTagWrapper<T> of(class_2378<T> registry, Map<class_2960, Collection<class_6880<T>>> vanillaTags) {
        return new VanillaTagWrapper<>(registry, vanillaTags);
    }

    public VanillaTagWrapper(class_2378<T> registry, Map<class_2960, Collection<class_6880<T>>> vanillaTags) {
        this.registry = registry;
        this.vanillaTags = vanillaTags;
    }

    public void add(class_2960 tag, class_6880<T> holder) {
        if (modifiedTags.contains(tag)) {
            vanillaTags.get(tag).add(holder);
            return;
        }

        Collection<class_6880<T>> existingHolders = vanillaTags.get(tag);
        Collection<class_6880<T>> newHolders = existingHolders == null ? new HashSet<>() : new HashSet<>(existingHolders);
        newHolders.add(holder);
        vanillaTags.put(tag, newHolders);
        modifiedTags.add(tag);
    }

    public boolean has(class_6862<T> tag) {
        return vanillaTags.containsKey(tag.comp_327());
    }

    public Collection<class_6880<T>> get(class_6862<T> tag) {
        return Collections.unmodifiableCollection(vanillaTags.getOrDefault(tag.comp_327(), Collections.emptyList()));
    }

    public Collection<class_6880<T>> get(class_2960 tag) {
        return Collections.unmodifiableCollection(vanillaTags.getOrDefault(tag, Collections.emptyList()));
    }

    public Set<class_2960> getTags(class_2960 entryId) {
        var key = class_5321.method_29179(registry.method_30517(), entryId);
        return registry.method_40264(key).map(this::getTags).orElse(Set.of());
    }

    public Set<class_2960> getTags(UnificationEntry<T> entry) {
        return getTags(entry.asHolderOrThrow());
    }

    public Set<class_2960> getTags(class_6880<T> holder) {
        if (holdersToTags == null) {
            holdersToTags = createInvertMap();
        }

        return holdersToTags.getOrDefault(holder, Set.of());
    }

    private Map<class_6880<T>, Set<class_2960>> createInvertMap() {
        Map<class_6880<T>, Set<class_2960>> map = new HashMap<>();

        for (var entry : vanillaTags.entrySet()) {
            for (class_6880<T> holder : entry.getValue()) {
                map.putIfAbsent(holder, new HashSet<>());
                map.get(holder).add(entry.getKey());
            }
        }

        return map;
    }

    public void seal() {
        for (class_2960 modifiedTag : modifiedTags) {
            Collection<class_6880<T>> holders = vanillaTags.get(modifiedTag);
            if (holders != null) {
                vanillaTags.put(modifiedTag, List.copyOf(holders));
            }
        }

        modifiedTags.clear();
        holdersToTags = null;
    }
}
