package com.almostreliable.unified.api.unification;

import org.jetbrains.annotations.Nullable;

import java.util.Collection;
import java.util.function.Predicate;
import net.minecraft.class_1792;
import net.minecraft.class_1799;
import net.minecraft.class_1856;
import net.minecraft.class_2960;
import net.minecraft.class_6862;
import net.minecraft.class_6880;
import net.minecraft.class_7923;

/**
 * Interface exposing composable unification information for a single unification config.
 * <p>
 * There exists one instance for each config.<br>
 * A unification lookup only exposes composable information. The composition of all lookups is used to check global
 * unification information. For example, when retrieving the replacement of a specific item. The composition can't
 * expose internal information such as mod priorities.
 * <p>
 * If a lookup with exposed configuration is required, use {@link UnificationSettings} instead.
 *
 * @since 1.0.0
 */
public interface UnificationLookup {

    /**
     * Returns all {@link class_6862}s used for the unification process of the config this lookup is for.
     * <p>
     * The returned collection will only contain tags that have their {@link Placeholders} replaced and have been
     * validated. All tags will be unique.
     *
     * @return the {@link class_6862}s used for the unification, or empty if no tags are used
     */
    Collection<class_6862<class_1792>> getTags();

    /**
     * Returns all {@link UnificationEntry}s for the given {@link class_6862}.
     * <p>
     * The returned collection will only contain entries if the provided {@link class_6862} is part of the config this
     * lookup is for.
     *
     * @param tag the {@link class_6862} to get the entries for
     * @return the {@link UnificationEntry}s for the {@link class_6862}, or empty if no {@link UnificationEntry}s are found
     */
    Collection<UnificationEntry<class_1792>> getTagEntries(class_6862<class_1792> tag);

    /**
     * Returns the {@link UnificationEntry} for the given item id.
     * <p>
     * If the config this lookup is for doesn't cover the {@link class_1792}, null is returned.
     *
     * @param item the item id to get the {@link UnificationEntry} for
     * @return the {@link UnificationEntry} for the item id, or null if no {@link UnificationEntry} is found
     */
    @Nullable
    UnificationEntry<class_1792> getItemEntry(class_2960 item);

    /**
     * Returns the {@link UnificationEntry} for the given {@link class_1792}.
     * <p>
     * If the config this lookup is for doesn't cover the {@link class_1792}, null is returned.
     *
     * @param item the {@link class_1792} to get the {@link UnificationEntry} for
     * @return the {@link UnificationEntry} for the {@link class_1792}, or null if no {@link UnificationEntry} is found
     */
    @Nullable
    default UnificationEntry<class_1792> getItemEntry(class_1792 item) {
        return getItemEntry(class_7923.field_41178.method_10221(item));
    }

    /**
     * Returns the {@link UnificationEntry} for the given item {@link class_6880}.
     * <p>
     * If the config this lookup is for doesn't cover the {@link class_1792}, null is returned.
     *
     * @param item the item {@link class_6880} to get the {@link UnificationEntry} for
     * @return the {@link UnificationEntry} for the item {@link class_6880}, or null if no {@link UnificationEntry} is found
     */
    @Nullable
    default UnificationEntry<class_1792> getItemEntry(class_6880<class_1792> item) {
        return getItemEntry(item.comp_349());
    }

    /**
     * Returns the relevant {@link class_6862} for the given item id.
     * <p>
     * Since an item can only have a single relevant tag, this method is guaranteed to return a single {@link class_6862} as
     * long as the config this lookup is for covers the {@link class_1792}.
     *
     * @param item the item id to get the relevant {@link class_6862} for
     * @return the relevant {@link class_6862} for the item id, or null if no relevant {@link class_6862} is found
     */
    @Nullable
    class_6862<class_1792> getRelevantItemTag(class_2960 item);

    /**
     * Returns the relevant {@link class_6862} for the given {@link class_1792}.
     * <p>
     * Since an item can only have a single relevant tag, this method is guaranteed to return a single {@link class_6862} as
     * long as the config this lookup is for covers the {@link class_1792}.
     *
     * @param item the {@link class_1792} to get the relevant {@link class_6862} for
     * @return the relevant {@link class_6862} for the {@link class_1792}, or null if no relevant {@link class_6862} is found
     */
    @Nullable
    default class_6862<class_1792> getRelevantItemTag(class_1792 item) {
        return getRelevantItemTag(class_7923.field_41178.method_10221(item));
    }

    /**
     * Returns the relevant {@link class_6862} for the given item {@link class_6880}.
     * <p>
     * Since an item can only have a single relevant tag, this method is guaranteed to return a single {@link class_6862} as
     * long as the config this lookup is for covers the {@link class_1792}.
     *
     * @param item the item {@link class_6880} to get the relevant {@link class_6862} for
     * @return the relevant {@link class_6862} for the item {@link class_6880}, or null if no relevant {@link class_6862} is found
     */
    @Nullable
    default class_6862<class_1792> getRelevantItemTag(class_6880<class_1792> item) {
        return getRelevantItemTag(item.comp_349());
    }

    /**
     * Returns the target item {@link UnificationEntry} for the given variant item id.
     * <p>
     * The target item describes the item with the highest priority among all variant items within a tag. It is used
     * to replace the variant items in the unification process.<br>
     * This method will return null if the config this lookup is for doesn't cover a unification tag that includes
     * the given item.
     * <p>
     * If the item is part of a stone variant, it will only check items within the same stone variant.
     *
     * @param item the variant item id to get the target {@link UnificationEntry} for
     * @return the target {@link UnificationEntry} for the variant item id, or null if no target
     * {@link UnificationEntry} is found
     */
    @Nullable
    UnificationEntry<class_1792> getVariantItemTarget(class_2960 item);

    /**
     * Returns the target item {@link UnificationEntry} for the given variant {@link class_1792}.
     * <p>
     * The target item describes the item with the highest priority among all variant items within a tag. It is used
     * to replace the variant items in the unification process.<br>
     * This method will return null if the config this lookup is for doesn't cover a unification tag that includes
     * the given item.
     * <p>
     * If the item is part of a stone variant, it will only check items within the same stone variant.
     *
     * @param item the variant {@link class_1792} to get the target {@link UnificationEntry} for
     * @return the target {@link UnificationEntry} for the variant {@link class_1792}, or null if no target
     * {@link UnificationEntry} is found
     */
    @Nullable
    default UnificationEntry<class_1792> getVariantItemTarget(class_1792 item) {
        return getVariantItemTarget(class_7923.field_41178.method_10221(item));
    }

    /**
     * Returns the target {@link UnificationEntry} for the given variant item {@link class_6880}.
     * <p>
     * The target item describes the item with the highest priority among all variant items within a tag. It is used
     * to replace the variant items in the unification process.<br>
     * This method will return null if the config this lookup is for doesn't cover a unification tag that includes
     * the given item.
     * <p>
     * If the item is part of a stone variant, it will only check items within the same stone variant.
     *
     * @param item the variant item {@link class_6880} to get the target {@link UnificationEntry} for
     * @return the target {@link UnificationEntry} for the variant item {@link class_6880}, or null if no target
     * {@link UnificationEntry} is found
     */
    @Nullable
    default UnificationEntry<class_1792> getVariantItemTarget(class_6880<class_1792> item) {
        return getVariantItemTarget(item.comp_349());
    }

    /**
     * Returns the target {@link UnificationEntry} for the given variant {@link UnificationEntry}.
     * <p>
     * The target item describes the item with the highest priority among all variant items within a {@link class_6862}.
     * It is used to replace the variant items in the unification process.<br>
     * This method will return null if the config this lookup is for doesn't cover a unification tag that includes
     * the given item.
     * <p>
     * If the item is part of a stone variant, it will only check items within the same stone variant.
     *
     * @param item the variant {@link UnificationEntry} to get the target {@link UnificationEntry} for
     * @return the target {@link UnificationEntry} for the variant {@link UnificationEntry}, or null if no target
     * {@link UnificationEntry} is found
     */
    @Nullable
    default UnificationEntry<class_1792> getVariantItemTarget(UnificationEntry<class_1792> item) {
        return getVariantItemTarget(item.asHolderOrThrow());
    }

    /**
     * Returns the target {@link UnificationEntry} for the given {@link class_6862} that matches the given filter.
     * <p>
     * The target item describes the item with the highest priority among all variant items within a {@link class_6862}.
     * It is used to replace the variant items in the unification process.<br>
     * If the config this lookup is for doesn't cover the {@link class_6862}, null is returned.
     *
     * @param tag the {@link class_6862} to get the target {@link UnificationEntry} for
     * @return the target {@link UnificationEntry} for the {@link class_6862}, or null if no target
     * {@link UnificationEntry} is found
     */
    @Nullable
    UnificationEntry<class_1792> getTagTargetItem(class_6862<class_1792> tag, Predicate<class_2960> itemFilter);

    /**
     * Returns the target {@link UnificationEntry} for the given {@link class_6862}.
     * <p>
     * The target item describes the item with the highest priority among all variant items within a {@link class_6862}.
     * It is used to replace the variant items in the unification process.<br>
     * If the config this lookup is for doesn't cover the {@link class_6862}, null is returned.
     *
     * @param tag the {@link class_6862} to get the target {@link UnificationEntry} for
     * @return the target {@link UnificationEntry} for the {@link class_6862}, or null if no target
     * {@link UnificationEntry} is found
     */
    @Nullable
    default UnificationEntry<class_1792> getTagTargetItem(class_6862<class_1792> tag) {
        return getTagTargetItem(tag, $ -> true);
    }

    /**
     * Returns whether the given {@link class_1799} is part of any tags the given {@link class_1856} points to.
     * <p>
     * To check this, this method fetches all unification tags of the {@link class_1792}s within the given {@link class_1856}
     * and checks whether the given {@link class_1799} is part of them.
     *
     * @param ingredient the {@link class_1856} to check to get the relevant tags for
     * @param item       the {@link class_1799} to check
     * @return whether the given {@link class_1799} is part of any tags the given {@link class_1856} points to
     */
    boolean isUnifiedIngredientItem(class_1856 ingredient, class_1799 item);
}
