package com.almostreliable.unified.unification;

import com.almostreliable.unified.AlmostUnifiedCommon;
import com.almostreliable.unified.api.unification.StoneVariants;
import com.almostreliable.unified.utils.VanillaTagWrapper;

import org.jetbrains.annotations.Nullable;

import java.util.Collection;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.regex.Pattern;
import net.minecraft.class_1792;
import net.minecraft.class_2248;
import net.minecraft.class_2960;
import net.minecraft.class_6862;
import net.minecraft.class_6880;
import net.minecraft.class_7924;

public final class StoneVariantsImpl implements StoneVariants {

    private static final Pattern ORE_TAG_PATTERN = Pattern.compile("(c:ores/.+|(minecraft|c):.+_ores)");

    private final Map<class_6862<class_1792>, Boolean> isOreTagCache = new HashMap<>();

    private final List<String> stoneVariants;
    private final Map<class_2960, String> itemToStoneVariant;

    private StoneVariantsImpl(Collection<String> stoneVariants, Map<class_2960, String> itemToStoneVariant) {
        this.stoneVariants = sortStoneVariants(stoneVariants);
        this.itemToStoneVariant = itemToStoneVariant;
    }

    public static StoneVariants create(Collection<String> stoneVariants, VanillaTagWrapper<class_1792> itemTags, VanillaTagWrapper<class_2248> blockTags) {
        Set<class_6862<class_1792>> stoneVariantItemTags = new HashSet<>();
        Set<class_6862<class_2248>> stoneVariantBlockTags = new HashSet<>();

        for (String stoneVariant : stoneVariants) {
            class_2960 id = class_2960.method_60655("c", "ores_in_ground/" + stoneVariant);
            stoneVariantItemTags.add(class_6862.method_40092(class_7924.field_41197, id));
            stoneVariantBlockTags.add(class_6862.method_40092(class_7924.field_41254, id));
        }

        var itemToStoneVariantTag = mapEntriesToStoneVariantTags(stoneVariantItemTags, itemTags);
        var blockToStoneVariantTag = mapEntriesToStoneVariantTags(stoneVariantBlockTags, blockTags);

        Map<class_2960, String> itemToStoneVariant = new HashMap<>();

        for (var entry : itemToStoneVariantTag.entrySet()) {
            class_2960 item = entry.getKey();
            class_6862<class_1792> tag = entry.getValue();
            String itemTagStoneVariant = getVariantFromStoneVariantTag(stoneVariants, tag);
            if (itemTagStoneVariant != null) {
                itemToStoneVariant.put(item, itemTagStoneVariant);
            }
        }

        for (var entry : blockToStoneVariantTag.entrySet()) {
            class_2960 item = entry.getKey();
            class_6862<class_2248> tag = entry.getValue();
            String blockTagStoneVariant = getVariantFromStoneVariantTag(stoneVariants, tag);
            if (blockTagStoneVariant == null) continue;

            String itemTagStoneVariant = itemToStoneVariant.get(item);
            if (itemTagStoneVariant == null || blockTagStoneVariant.length() > itemTagStoneVariant.length()) {
                itemToStoneVariant.put(item, blockTagStoneVariant);
            }
        }

        return new StoneVariantsImpl(stoneVariants, itemToStoneVariant);
    }

    /**
     * Maps all entries of a stone variant tag to its respective stone variant tag.
     *
     * @param stoneVariantTags the stone variant tags
     * @param tags             the vanilla tag wrapper to get the tag entries from
     * @param <T>              the tag type
     * @return the entry to stone variant tag mapping
     */
    private static <T> Map<class_2960, class_6862<T>> mapEntriesToStoneVariantTags(Set<class_6862<T>> stoneVariantTags, VanillaTagWrapper<T> tags) {
        Map<class_2960, class_6862<T>> idToStoneVariantTag = new HashMap<>();

        for (var stoneVariantTag : stoneVariantTags) {
            for (var holder : tags.get(stoneVariantTag)) {
                class_2960 id = holder
                    .method_40230()
                    .orElseThrow(() -> new IllegalStateException("Tag is not bound for holder " + holder))
                    .method_29177();

                var oldTag = idToStoneVariantTag.put(id, stoneVariantTag);
                if (oldTag != null) {
                    AlmostUnifiedCommon.LOGGER.error(
                        "{} is bound to multiple stone variant tags: {} and {}",
                        id,
                        oldTag,
                        stoneVariantTag
                    );
                }
            }
        }

        return idToStoneVariantTag;
    }

    /**
     * Returns the variant of a stone variant tag.
     * <p>
     * Example: {@code c:ores_in_ground/deepslate} -> {@code deepslate}
     *
     * @param stoneVariants the available stone variants
     * @param tag           the stone variant tag
     * @return the stone variant, or null if the stone strata is not a configured variant
     */
    @Nullable
    private static String getVariantFromStoneVariantTag(Collection<String> stoneVariants, class_6862<?> tag) {
        String tagString = tag.comp_327().toString();
        int i = tagString.lastIndexOf('/');
        String stoneVariant = tagString.substring(i + 1);
        if (!stoneVariants.contains(stoneVariant)) {
            return null;
        }

        return stoneVariant.equals("stone") ? "" : stoneVariant;
    }

    @Override
    public String getStoneVariant(class_2960 item) {
        return itemToStoneVariant.computeIfAbsent(item, this::computeStoneVariant);
    }

    @Override
    public boolean isOreTag(class_6862<class_1792> tag) {
        return isOreTagCache.computeIfAbsent(tag, t -> ORE_TAG_PATTERN.matcher(t.comp_327().toString()).matches());
    }

    /**
     * Returns a list of all stone variants sorted from longest to shortest.
     * <p>
     * This is required to ensure that the longest variant is returned first and no sub-matches happen.<br>
     * Example: "nether" and "blue_nether" would both match "nether" if the list is not sorted.
     *
     * @param stoneVariants the stone variants to sort
     * @return the sorted stone variants
     */
    private static List<String> sortStoneVariants(Collection<String> stoneVariants) {
        return stoneVariants.stream().sorted(Comparator.comparingInt(String::length).reversed()).toList();
    }

    /**
     * Implementation logic for caching in {@link #getStoneVariant(class_2960)}.
     *
     * @param item the item to get the stone variant of
     * @return the stone variant of the item
     */
    private String computeStoneVariant(class_2960 item) {
        for (String stoneVariant : stoneVariants) {
            if (item.method_12832().contains(stoneVariant + "_") || item.method_12832().endsWith("_" + stoneVariant)) {
                return stoneVariant.equals("stone") ? "" : stoneVariant;
            }
        }

        return "";
    }
}
