package com.almostreliable.unified.compat.viewer;

import com.almostreliable.unified.unification.recipe.RecipeLink;
import com.almostreliable.unified.utils.Utils;

import com.google.common.collect.ImmutableMap;
import com.google.gson.JsonArray;
import com.google.gson.JsonObject;
import com.mojang.serialization.Codec;
import com.mojang.serialization.MapCodec;
import com.mojang.serialization.codecs.RecordCodecBuilder;

import org.jetbrains.annotations.Nullable;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import net.minecraft.class_1799;
import net.minecraft.class_1860;
import net.minecraft.class_1865;
import net.minecraft.class_1937;
import net.minecraft.class_2960;
import net.minecraft.class_3956;
import net.minecraft.class_7225;
import net.minecraft.class_9129;
import net.minecraft.class_9139;
import net.minecraft.class_9695;

/**
 * This recipe is used to track which recipes were unified. It is NOT used for crafting.
 * Each tracker will hold one namespace with a list of recipes that were unified for it.
 */
public record ClientRecipeTracker(String namespace, Map<class_2960, ClientRecipeLink> recipes)
    implements class_1860<class_9695> {

    public static final class_2960 ID = Utils.getRL("client_recipe_tracker");
    public static final String RECIPES = "recipes";
    public static final String NAMESPACE = "namespace";
    public static final int UNIFIED_FLAG = 1;
    public static final int DUPLICATE_FLAG = 2;
    public static final class_1865<ClientRecipeTracker> SERIALIZER = new Serializer();
    public static final class_3956<ClientRecipeTracker> TYPE = new class_3956<>() {
        @Override
        public String toString() {
            return ID.method_12832();
        }
    };


    /**
     * Creates a raw string representation.
     *
     * @param isUnified   Whether the recipe was unified.
     * @param isDuplicate Whether the recipe had duplicates.
     * @param idPath      The path of the recipe.
     * @return String representation as: `flag$idPath`
     */
    private static String createRaw(boolean isUnified, boolean isDuplicate, String idPath) {
        int flag = 0;
        if (isUnified) flag |= UNIFIED_FLAG;
        if (isDuplicate) flag |= DUPLICATE_FLAG;
        return flag + "$" + idPath;
    }

    //<editor-fold defaultstate="collapsed" desc="Default recipe stuff. Ignore this. Forget this.">
    @Override
    public boolean method_8115(class_9695 recipeInput, class_1937 level) {
        return false;
    }

    @Override
    public class_1799 method_8116(class_9695 recipeInput, class_7225.class_7874 provider) {
        return class_1799.field_8037;
    }

    @Override
    public boolean method_8113(int width, int height) {
        return false;
    }

    @Override
    public class_1799 method_8110(class_7225.class_7874 provider) {
        return class_1799.field_8037;
    }
    //</editor-fold>

    @Override
    public class_1865<?> method_8119() {
        return SERIALIZER;
    }

    @Override
    public class_3956<?> method_17716() {
        return TYPE;
    }

    private void add(ClientRecipeLink clientRecipeLink) {
        recipes.put(clientRecipeLink.id(), clientRecipeLink);
    }

    @Nullable
    public ClientRecipeLink getLink(class_2960 recipeId) {
        return recipes.get(recipeId);
    }

    public record ClientRecipeLink(class_2960 id, boolean isUnified, boolean isDuplicate) {}


    public List<String> getLinkStrings() {
        return recipes.values().stream().map(l -> createRaw(l.isUnified, l.isDuplicate, l.id.method_12832())).toList();
    }

    public static class Serializer implements class_1865<ClientRecipeTracker> {

        /**
         * Codec for the recipe tracker. The recipe will look like this:
         * <pre>
         * {@code
         * {
         *      "type": "almostunified:client_recipe_tracker",
         *      "namespace": "minecraft", // The namespace of the recipes.
         *      "recipes": [
         *          "flag$recipePath",
         *          "flag$recipe2Path",
         *          ...
         *          "flag$recipeNPath"
         *      ]
         * }
         * }
         * </pre>
         */
        public static final MapCodec<ClientRecipeTracker> CODEC = RecordCodecBuilder.mapCodec(instance -> instance
            .group(
                Codec.STRING.fieldOf("namespace").forGetter(ClientRecipeTracker::namespace),
                Codec.list(Codec.STRING).fieldOf("recipes").forGetter(ClientRecipeTracker::getLinkStrings)
            )
            .apply(instance, Serializer::of));


        public static final class_9139<class_9129, ClientRecipeTracker> STREAM_CODEC = new class_9139<>() {
            @Override
            public ClientRecipeTracker decode(class_9129 buffer) {
                int size = buffer.readInt();
                String namespace = buffer.method_19772();

                ImmutableMap.Builder<class_2960, ClientRecipeLink> builder = ImmutableMap.builder();
                for (int i = 0; i < size; i++) {
                    String raw = buffer.method_19772();
                    ClientRecipeLink clientRecipeLink = parseRaw(namespace, raw);
                    builder.put(clientRecipeLink.id(), clientRecipeLink);
                }

                return new ClientRecipeTracker(namespace, builder.build());
            }

            /**
             * Writes the tracker to the buffer. The namespace is written separately to save some bytes.
             * Buffer output will look like:
             * <pre>
             *     size
             *     namespace
             *     flag$recipePath
             *     flag$recipe2Path
             *     ...
             *     flag$recipeNPath
             * </pre>
             *
             * @param buffer The buffer to write to
             * @param recipe The recipe to write
             */
            @Override
            public void encode(class_9129 buffer, ClientRecipeTracker recipe) {
                buffer.method_53002(recipe.recipes.size());
                buffer.method_10814(recipe.namespace);
                for (ClientRecipeLink clientRecipeLink : recipe.recipes.values()) {
                    String raw = createRaw(clientRecipeLink.isUnified(),
                        clientRecipeLink.isDuplicate(),
                        clientRecipeLink.id().method_12832());
                    buffer.method_10814(raw);
                }
            }
        };

        @Override
        public MapCodec<ClientRecipeTracker> method_53736() {
            return CODEC;
        }

        @Override
        public class_9139<class_9129, ClientRecipeTracker> method_56104() {
            return STREAM_CODEC;
        }

        private static ClientRecipeTracker of(String namespace, List<String> recipes) {
            ImmutableMap.Builder<class_2960, ClientRecipeLink> builder = ImmutableMap.builder();

            for (String recipe : recipes) {
                ClientRecipeLink link = parseRaw(namespace, recipe);
                builder.put(link.id(), link);
            }

            return new ClientRecipeTracker(namespace, builder.build());
        }

        /**
         * Creates a {@link ClientRecipeLink} from a raw string for the given namespace.
         *
         * @param namespace The namespace to use.
         * @param raw       The raw string.
         * @return The client sided recipe link.
         */
        public static ClientRecipeLink parseRaw(String namespace, String raw) {
            String[] split = raw.split("\\$", 2);
            int flag = Integer.parseInt(split[0]);
            boolean isUnified = (flag & UNIFIED_FLAG) != 0;
            boolean isDuplicate = (flag & DUPLICATE_FLAG) != 0;
            return new ClientRecipeLink(
                class_2960.method_60655(namespace, split[1]),
                isUnified,
                isDuplicate
            );
        }
    }

    public static class RawBuilder {

        private final Map<String, JsonArray> recipesByNamespace = new HashMap<>();

        public void add(RecipeLink recipe) {
            class_2960 recipeId = recipe.getId();
            JsonArray array = recipesByNamespace.computeIfAbsent(recipeId.method_12836(), k -> new JsonArray());
            array.add(createRaw(recipe.isUnified(), recipe.hasDuplicateLink(), recipeId.method_12832()));
        }

        /**
         * Creates a map with the namespace as key and the JSON recipe.
         * These recipes are used later in {@link Serializer}
         *
         * @return The map with the namespace as key and the JSON recipe.
         */
        public Map<class_2960, JsonObject> compute() {
            Map<class_2960, JsonObject> result = new HashMap<>();
            recipesByNamespace.forEach((namespace, recipes) -> {
                JsonObject json = new JsonObject();
                json.addProperty("type", ID.toString());
                json.addProperty(NAMESPACE, namespace);
                json.add(RECIPES, recipes);
                result.put(Utils.getRL(namespace), json);
            });
            return result;
        }
    }
}
