package com.almostreliable.unified.utils;

import java.util.*;
import java.util.Map.Entry;
import java.util.function.Predicate;
import net.minecraft.class_1792;
import net.minecraft.class_2248;
import net.minecraft.class_2378;
import net.minecraft.class_2960;
import net.minecraft.class_3503;
import net.minecraft.class_5321;
import net.minecraft.class_6862;
import net.minecraft.class_6880;
import net.minecraft.class_7923;
import net.minecraft.class_7924;

public class TagMap<T> {

    private final Map<UnifyTag<T>, Set<class_2960>> tagsToEntries = new HashMap<>();
    private final Map<class_2960, Set<UnifyTag<T>>> entriesToTags = new HashMap<>();

    protected TagMap() {}

    /**
     * Creates an item tag map from a set of item unify tags.
     * <p>
     * This should only be used for client-side tag maps or for tests.<br>
     * It requires the registry to be loaded in order to validate the tags
     * and fetch the holder from it.
     * <p>
     * For the server, use {@link #createFromItemTags(Map)} instead.
     *
     * @param unifyTags The unify tags.
     * @return A new tag map.
     */
    public static TagMap<class_1792> create(Set<UnifyTag<class_1792>> unifyTags) {
        TagMap<class_1792> tagMap = new TagMap<>();

        unifyTags.forEach(ut -> {
            class_6862<class_1792> asTagKey = class_6862.method_40092(class_7924.field_41197, ut.location());
            class_7923.field_41178.method_40286(asTagKey).forEach(holder -> {
                class_2960 key = class_7923.field_41178.method_10221(holder.comp_349());
                tagMap.put(ut, key);
            });
        });

        return tagMap;
    }

    /**
     * Creates an item tag map from the vanilla item tag collection passed by the {@link class_3503}.
     * <p>
     * This should only be used on the server.<br>
     * This tag map should later be filtered by using {@link #filtered(Predicate, Predicate)}.
     * <p>
     * For the client, use {@link #create(Set)} instead.
     *
     * @param tags The vanilla item tag collection.
     * @return A new item tag map.
     */
    public static TagMap<class_1792> createFromItemTags(Map<class_2960, Collection<class_6880<class_1792>>> tags) {
        TagMap<class_1792> tagMap = new TagMap<>();

        for (var entry : tags.entrySet()) {
            UnifyTag<class_1792> unifyTag = UnifyTag.item(entry.getKey());
            fillEntries(tagMap, entry.getValue(), unifyTag, class_7923.field_41178);
        }

        return tagMap;
    }

    /**
     * Creates a block tag map from the vanilla block tag collection passed by the {@link class_3503}.
     * <p>
     * This should only be used on the server.
     *
     * @param tags The vanilla block tag collection.
     * @return A new block tag map.
     */
    public static TagMap<class_2248> createFromBlockTags(Map<class_2960, Collection<class_6880<class_2248>>> tags) {
        TagMap<class_2248> tagMap = new TagMap<>();

        for (var entry : tags.entrySet()) {
            UnifyTag<class_2248> unifyTag = UnifyTag.block(entry.getKey());
            fillEntries(tagMap, entry.getValue(), unifyTag, class_7923.field_41175);
        }

        return tagMap;
    }

    /**
     * Unwrap all holders, verify them and put them into the tag map.
     *
     * @param tagMap   The tag map to fill.
     * @param holders  The holders to unwrap.
     * @param unifyTag The unify tag to use.
     * @param registry The registry to use.
     */
    private static <T> void fillEntries(TagMap<T> tagMap, Collection<class_6880<T>> holders, UnifyTag<T> unifyTag, class_2378<T> registry) {
        for (var holder : holders) {
            holder
                    .method_40230()
                    .map(class_5321::method_29177)
                    .filter(registry::method_10250)
                    .ifPresent(id -> tagMap.put(unifyTag, id));
        }
    }

    /**
     * Creates a filtered tag map copy.
     *
     * @param tagFilter   A filter to determine which tags to include.
     * @param entryFilter A filter to determine which entries to include.
     * @return A filtered copy of this tag map.
     */
    public TagMap<T> filtered(Predicate<UnifyTag<T>> tagFilter, Predicate<class_2960> entryFilter) {
        TagMap<T> tagMap = new TagMap<>();

        tagsToEntries.forEach((tag, items) -> {
            if (!tagFilter.test(tag)) {
                return;
            }
            items.stream().filter(entryFilter).forEach(item -> tagMap.put(tag, item));
        });

        return tagMap;
    }

    public int tagSize() {
        return tagsToEntries.size();
    }

    public int itemSize() {
        return entriesToTags.size();
    }

    public Set<class_2960> getEntriesByTag(UnifyTag<T> tag) {
        return Collections.unmodifiableSet(tagsToEntries.getOrDefault(tag, Collections.emptySet()));
    }

    public Set<UnifyTag<T>> getTagsByEntry(class_2960 entry) {
        return Collections.unmodifiableSet(entriesToTags.getOrDefault(entry, Collections.emptySet()));
    }

    public Set<UnifyTag<T>> getTags() {
        return Collections.unmodifiableSet(tagsToEntries.keySet());
    }

    /**
     * Helper function to build a relationship between a tag and an entry.
     * <p>
     * If the entries don't exist in the internal maps yet, they will be created. That means
     * it needs to be checked whether the tag or entry is valid before calling this method.
     *
     * @param tag   The tag.
     * @param entry The entry.
     */
    protected void put(UnifyTag<T> tag, class_2960 entry) {
        tagsToEntries.computeIfAbsent(tag, k -> new HashSet<>()).add(entry);
        entriesToTags.computeIfAbsent(entry, k -> new HashSet<>()).add(tag);
    }
}
