package com.almostreliable.unified.utils;

import com.almostreliable.unified.AlmostUnified;
import com.almostreliable.unified.api.StoneStrataHandler;
import com.almostreliable.unified.config.UnifyConfig;
import javax.annotation.Nullable;
import net.minecraft.class_1792;
import net.minecraft.class_1799;
import net.minecraft.class_1856;
import net.minecraft.class_2960;
import net.minecraft.class_6862;
import net.minecraft.class_7923;
import net.minecraft.class_7924;
import java.util.*;
import java.util.function.Predicate;

public class ReplacementMap {

    private final UnifyConfig unifyConfig;
    private final TagMap<class_1792> tagMap;
    private final StoneStrataHandler stoneStrataHandler;
    private final TagOwnerships tagOwnerships;
    private final Set<class_2960> warnings;

    public ReplacementMap(UnifyConfig unifyConfig, TagMap<class_1792> tagMap, StoneStrataHandler stoneStrataHandler, TagOwnerships tagOwnerships) {
        this.tagMap = tagMap;
        this.unifyConfig = unifyConfig;
        this.stoneStrataHandler = stoneStrataHandler;
        this.tagOwnerships = tagOwnerships;
        this.warnings = new HashSet<>();
    }

    @Nullable
    public UnifyTag<class_1792> getPreferredTagForItem(class_2960 item) {
        Collection<UnifyTag<class_1792>> tags = tagMap.getTagsByEntry(item);

        if (tags.isEmpty()) {
            return null;
        }

        if (tags.size() > 1 && !warnings.contains(item)) {
            AlmostUnified.LOG.warn(
                    "Item '{}' has multiple preferred tags '{}' for recipe replacement. This needs to be manually fixed by the user.",
                    item,
                    tags.stream().map(UnifyTag::location).toList()
            );
            warnings.add(item);
        }

        return tags.iterator().next();
    }

    @Nullable
    public class_2960 getReplacementForItem(class_2960 item) {
        UnifyTag<class_1792> t = getPreferredTagForItem(item);
        if (t == null) {
            return null;
        }

        if (stoneStrataHandler.isStoneStrataTag(t)) {
            String stone = stoneStrataHandler.getStoneStrata(item);
            return getPreferredItemForTag(t, i -> stone.equals(stoneStrataHandler.getStoneStrata(i)));
        }

        return getPreferredItemForTag(t, i -> true);
    }

    @Nullable
    public class_2960 getPreferredItemForTag(UnifyTag<class_1792> tag, Predicate<class_2960> itemFilter) {
        var tagToLookup = tagOwnerships.getOwnerByTag(tag);
        if (tagToLookup == null) tagToLookup = tag;

        List<class_2960> items = tagMap
                .getEntriesByTag(tagToLookup)
                .stream()
                .filter(itemFilter)
                // Helps us to get the clean stone variant first in case of a stone strata tag
                .sorted(Comparator.comparingInt(value -> value.toString().length()))
                .toList();

        if (items.isEmpty()) return null;

        class_2960 overrideItem = getOverrideForTag(tagToLookup, items);
        if (overrideItem != null) {
            return overrideItem;
        }

        for (String modPriority : unifyConfig.getModPriorities()) {
            class_2960 item = findItemByNamespace(items, modPriority);
            if (item != null) return item;
        }

        return null;
    }

    /**
     * Gets all unify tags of the items within the given ingredient and checks
     * whether the given item is in one of those tags.
     *
     * @param ingred The ingredient to get the unify tags from.
     * @param item   The item to check.
     * @return Whether the item is in one of the unify tags of the ingredient.
     */
    public boolean isItemInUnifiedIngredient(class_1856 ingred, class_1799 item) {
        Set<UnifyTag<class_1792>> checkedTags = new HashSet<>();

        for (class_1799 ingredItem : ingred.method_8105()) {
            class_2960 itemId = class_7923.field_41178.method_10221(ingredItem.method_7909());

            var preferredTag = getPreferredTagForItem(itemId);
            if (preferredTag == null || checkedTags.contains(preferredTag)) continue;
            checkedTags.add(preferredTag);

            var preferredTagKey = class_6862.method_40092(class_7924.field_41197, preferredTag.location());
            if (item.method_31573(preferredTagKey)) {
                return true;
            }
        }

        return false;
    }

    @Nullable
    private class_2960 getOverrideForTag(UnifyTag<class_1792> tag, List<class_2960> items) {
        String priorityOverride = unifyConfig.getPriorityOverrides().get(tag.location());
        if (priorityOverride != null) {
            class_2960 item = findItemByNamespace(items, priorityOverride);
            if (item != null) return item;
            AlmostUnified.LOG.warn(
                    "Priority override mod '{}' for tag '{}' does not contain a valid item. Falling back to default priority.",
                    priorityOverride,
                    tag.location());
        }
        return null;
    }

    @Nullable
    private class_2960 findItemByNamespace(List<class_2960> items, String namespace) {
        for (class_2960 item : items) {
            if (item.method_12836().equals(namespace)) {
                return item;
            }
        }
        return null;
    }

    public TagOwnerships getTagOwnerships() {
        return tagOwnerships;
    }
}
