package com.almostreliable.unified.config;

import com.almostreliable.unified.AlmostUnifiedPlatform;
import com.almostreliable.unified.AlmostUnifiedPlatform.Platform;
import com.almostreliable.unified.recipe.RecipeLink;
import com.almostreliable.unified.utils.JsonCompare;
import com.google.gson.JsonObject;
import java.util.*;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import net.minecraft.class_2960;

public class DuplicationConfig extends Config {
    public static final String NAME = "duplicates";

    private final JsonCompare.CompareSettings defaultRules;
    private final LinkedHashMap<class_2960, JsonCompare.CompareSettings> overrideRules;
    private final Set<Pattern> ignoreRecipeTypes;
    private final Set<Pattern> ignoreRecipes;
    private final boolean strictMode;
    private final Map<class_2960, Boolean> ignoredRecipeTypesCache;

    public DuplicationConfig(JsonCompare.CompareSettings defaultRules, LinkedHashMap<class_2960, JsonCompare.CompareSettings> overrideRules, Set<Pattern> ignoreRecipeTypes, Set<Pattern> ignoreRecipes, boolean strictMode) {
        this.defaultRules = defaultRules;
        this.overrideRules = overrideRules;
        this.ignoreRecipeTypes = ignoreRecipeTypes;
        this.ignoreRecipes = ignoreRecipes;
        this.strictMode = strictMode;
        this.ignoredRecipeTypesCache = new HashMap<>();
    }

    public boolean shouldIgnoreRecipe(RecipeLink recipe) {
        if (isRecipeTypeIgnored(recipe)) {
            return true;
        }

        for (Pattern ignoreRecipePattern : ignoreRecipes) {
            if (ignoreRecipePattern.matcher(recipe.getId().toString()).matches()) {
                return true;
            }
        }

        return false;
    }

    /**
     * Checks if the recipe type is ignored. This is cached to avoid having to recompute the regex for every recipe.
     *
     * @param recipe The recipe to check
     * @return True if the recipe type is ignored, false otherwise
     */
    private boolean isRecipeTypeIgnored(RecipeLink recipe) {
        class_2960 type = recipe.getType();
        Boolean ignored = ignoredRecipeTypesCache.get(type);

        if (ignored == null) {
            ignored = computeIsRecipeTypeIgnored(type.toString());
            ignoredRecipeTypesCache.put(type, ignored);
        }

        return ignored;
    }

    private boolean computeIsRecipeTypeIgnored(String recipeType) {
        for (Pattern ignorePattern : ignoreRecipeTypes) {
            if (ignorePattern.matcher(recipeType).matches()) {
                return true;
            }
        }

        return false;
    }

    public JsonCompare.CompareSettings getCompareSettings(class_2960 type) {
        return overrideRules.getOrDefault(type, defaultRules);
    }

    public boolean isStrictMode() {
        return strictMode;
    }

    public JsonCompare.CompareContext getCompareContext(RecipeLink recipe) {
        JsonCompare.CompareSettings compareSettings = getCompareSettings(recipe.getType());
        return JsonCompare.CompareContext.create(compareSettings, recipe.getActual());
    }

    public void clearCache() {
        ignoredRecipeTypesCache.clear();
    }

    public static class Serializer extends Config.Serializer<DuplicationConfig> {
        public static final String DEFAULT_DUPLICATE_RULES = "defaultDuplicateRules";
        public static final String OVERRIDE_DUPLICATE_RULES = "overrideDuplicateRules";
        public static final String IGNORED_RECIPE_TYPES = "ignoredRecipeTypes";
        public static final String IGNORED_RECIPES = "ignoredRecipes";
        public static final String STRICT_MODE = "strictMode";

        @Override
        public DuplicationConfig deserialize(JsonObject json) {
            var platform = AlmostUnifiedPlatform.INSTANCE.getPlatform();
            Set<Pattern> ignoreRecipeTypes = deserializePatterns(json,
                    IGNORED_RECIPE_TYPES,
                    Defaults.getIgnoredRecipeTypes(platform));
            Set<Pattern> ignoreRecipes = deserializePatterns(json, IGNORED_RECIPES, List.of());

            JsonCompare.CompareSettings defaultRules = safeGet(() -> createCompareSet(json.getAsJsonObject(
                            DEFAULT_DUPLICATE_RULES)),
                    Defaults.getDefaultDuplicateRules(platform));
            LinkedHashMap<class_2960, JsonCompare.CompareSettings> overrideRules = safeGet(() ->
                    getOverrideRules(json), Defaults.getDefaultDuplicateOverrides(platform));
            boolean strictMode = safeGet(() -> json.get(STRICT_MODE).getAsBoolean(), false);

            return new DuplicationConfig(defaultRules, overrideRules, ignoreRecipeTypes, ignoreRecipes, strictMode);
        }

        // Extracted as method because `safeGet` couldn't cast the type... Seems to be an old SDK bug :-)
        // https://bugs.openjdk.org/browse/JDK-8324860
        private LinkedHashMap<class_2960, JsonCompare.CompareSettings> getOverrideRules(JsonObject json) {
            return json
                    .getAsJsonObject(OVERRIDE_DUPLICATE_RULES)
                    .entrySet()
                    .stream()
                    .collect(Collectors.toMap(entry -> new class_2960(entry.getKey()),
                            entry -> createCompareSet(entry.getValue().getAsJsonObject()),
                            (a, b) -> b,
                            LinkedHashMap::new));
        }

        private JsonCompare.CompareSettings createCompareSet(JsonObject rules) {
            JsonCompare.CompareSettings result = new JsonCompare.CompareSettings();
            result.deserialize(rules);
            return result;
        }

        @Override
        public JsonObject serialize(DuplicationConfig config) {
            JsonObject json = new JsonObject();

            serializePatterns(json, IGNORED_RECIPE_TYPES, config.ignoreRecipeTypes);
            serializePatterns(json, IGNORED_RECIPES, config.ignoreRecipes);
            json.add(DEFAULT_DUPLICATE_RULES, config.defaultRules.serialize());
            JsonObject overrides = new JsonObject();
            config.overrideRules.forEach((rl, compareSettings) -> {
                overrides.add(rl.toString(), compareSettings.serialize());
            });
            json.add(OVERRIDE_DUPLICATE_RULES, overrides);
            json.addProperty(STRICT_MODE, false);

            return json;
        }

    }
}
