/*
 * BluSunrize
 * Copyright (c) 2017
 *
 * This code is licensed under "Blu's License of Common Sense"
 * Details can be found in the license file in the root folder of this project
 */

package blusunrize.immersiveengineering.common.util;

import blusunrize.immersiveengineering.ImmersiveEngineering;
import blusunrize.immersiveengineering.api.ApiUtils;
import blusunrize.immersiveengineering.api.DirectionalBlockPos;
import blusunrize.immersiveengineering.api.Lib;
import blusunrize.immersiveengineering.api.crafting.IngredientStack;
import blusunrize.immersiveengineering.common.util.inventory.IIEInventory;
import com.google.common.base.Charsets;
import com.google.common.base.Optional;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Lists;
import com.google.common.io.Resources;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonParseException;
import crafttweaker.api.block.IBlock;
import net.minecraft.advancements.Advancement;
import net.minecraft.advancements.AdvancementManager;
import net.minecraft.advancements.PlayerAdvancements;
import net.minecraft.block.*;
import net.minecraft.block.material.Material;
import net.minecraft.block.properties.IProperty;
import net.minecraft.block.state.BlockFaceShape;
import net.minecraft.block.state.IBlockState;
import net.minecraft.client.resources.I18n;
import net.minecraft.entity.EntityLivingBase;
import net.minecraft.entity.item.EntityItem;
import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.entity.player.EntityPlayerMP;
import net.minecraft.init.Blocks;
import net.minecraft.init.Items;
import net.minecraft.inventory.Container;
import net.minecraft.inventory.IInventory;
import net.minecraft.inventory.InventoryCrafting;
import net.minecraft.item.Item;
import net.minecraft.item.ItemDye;
import net.minecraft.item.ItemStack;
import net.minecraft.item.crafting.CraftingManager;
import net.minecraft.item.crafting.IRecipe;
import net.minecraft.nbt.NBTBase;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.nbt.NBTTagList;
import net.minecraft.tileentity.TileEntity;
import net.minecraft.util.DamageSource;
import net.minecraft.util.EnumFacing;
import net.minecraft.util.EnumFacing.Axis;
import net.minecraft.util.NonNullList;
import net.minecraft.util.ResourceLocation;
import net.minecraft.util.math.*;
import net.minecraft.world.IBlockAccess;
import net.minecraft.world.World;
import net.minecraft.world.WorldServer;
import net.minecraft.world.storage.loot.*;
import net.minecraft.world.storage.loot.conditions.LootCondition;
import net.minecraft.world.storage.loot.conditions.LootConditionManager;
import net.minecraft.world.storage.loot.functions.LootFunction;
import net.minecraft.world.storage.loot.functions.LootFunctionManager;
import net.minecraftforge.common.property.IExtendedBlockState;
import net.minecraftforge.common.property.IUnlistedProperty;
import net.minecraftforge.common.util.Constants;
import net.minecraftforge.fluids.*;
import net.minecraftforge.fluids.capability.CapabilityFluidHandler;
import net.minecraftforge.fluids.capability.IFluidHandler;
import net.minecraftforge.fml.common.Loader;
import net.minecraftforge.fml.common.ModContainer;
import net.minecraftforge.items.CapabilityItemHandler;
import net.minecraftforge.items.IItemHandler;
import net.minecraftforge.items.ItemHandlerHelper;
import net.minecraftforge.oredict.OreDictionary;

import javax.annotation.Nullable;
import java.io.IOException;
import java.lang.reflect.Method;
import java.net.URL;
import java.text.DecimalFormat;
import java.util.*;
import java.util.function.Consumer;

import static java.lang.Math.min;

public class Utils
{
	public static final Random RAND = new Random();

	public static boolean compareToOreName(ItemStack stack, String oreName)
	{
		if(!ApiUtils.isExistingOreName(oreName))
			return false;
		ItemStack comp = copyStackWithAmount(stack, 1);
		List<ItemStack> s = OreDictionary.getOres(oreName);
		for(ItemStack st:s)
			if(OreDictionary.itemMatches(st, comp, false))
				return true;
		return false;
	}
	public static boolean stackMatchesObject(ItemStack stack, Object o)
	{
		return stackMatchesObject(stack, o, false);
	}
	public static boolean stackMatchesObject(ItemStack stack, Object o, boolean checkNBT)
	{
		if(o instanceof ItemStack)
			return OreDictionary.itemMatches((ItemStack)o, stack, false) && (!checkNBT || ((ItemStack)o).func_77952_i()==OreDictionary.WILDCARD_VALUE || Utils.compareItemNBT((ItemStack)o, stack));
		else if(o instanceof Collection)
		{
			for(Object io : (Collection)o)
				if(io instanceof ItemStack && OreDictionary.itemMatches((ItemStack)io, stack, false) && (!checkNBT || ((ItemStack)io).func_77952_i()==OreDictionary.WILDCARD_VALUE || Utils.compareItemNBT((ItemStack)io, stack)))
					return true;
		} else if(o instanceof IngredientStack)
			return ((IngredientStack)o).matchesItemStack(stack);
		else if(o instanceof ItemStack[])
		{
			for(ItemStack io : (ItemStack[])o)
				if(OreDictionary.itemMatches(io, stack, false) && (!checkNBT || io.func_77952_i()==OreDictionary.WILDCARD_VALUE || Utils.compareItemNBT(io, stack)))
					return true;
		} else if(o instanceof FluidStack)
		{
			FluidStack fs = FluidUtil.getFluidContained(stack);
			return fs != null && fs.containsFluid((FluidStack)o);
		}
		else if(o instanceof String)
			return compareToOreName(stack, (String)o);
		return false;
	}
	public static boolean compareItemNBT(ItemStack stack1, ItemStack stack2)
	{
		if((stack1.func_190926_b()) != (stack2.func_190926_b()))
			return false;
		boolean empty1 = (stack1.func_77978_p()==null||stack1.func_77978_p().func_82582_d());
		boolean empty2 = (stack2.func_77978_p()==null||stack2.func_77978_p().func_82582_d());
		if(empty1!=empty2)
			return false;
		if(!empty1 && !stack1.func_77978_p().equals(stack2.func_77978_p()))
			return false;
		return stack1.areCapsCompatible(stack2);
	}

	public static boolean canCombineArrays(ItemStack[] stacks, ItemStack[] target)
	{
		HashSet<IngredientStack> inputSet = new HashSet();
		for(ItemStack s : stacks)
			inputSet.add(new IngredientStack(s));
		for(ItemStack t : target)
		{
			int size = t.func_190916_E();
			Iterator<IngredientStack> it = inputSet.iterator();
			while(it.hasNext())
			{
				IngredientStack in = it.next();
				if(in.matchesItemStackIgnoringSize(t))
				{
					int taken = Math.min(size, in.inputSize);
					size -= taken;
					in.inputSize -= taken;
					if(in.inputSize<=0)
						it.remove();
					if(size<=0)
						break;
				}
			}
			if(size>0)
				return false;
		}
		return true;
	}
	public static ItemStack copyStackWithAmount(ItemStack stack, int amount)
	{
		if(stack.func_190926_b())
			return ItemStack.field_190927_a;
		ItemStack s2 = stack.func_77946_l();
		s2.func_190920_e(amount);
		return s2;
	}
	public static String[] dyeNames = {"Black","Red","Green","Brown","Blue","Purple","Cyan","LightGray","Gray","Pink","Lime","Yellow","LightBlue","Magenta","Orange","White"};
	public static int getDye(ItemStack stack)
	{
		if(stack.func_190926_b())
			return -1;
		if(stack.func_77973_b().equals(Items.field_151100_aR))
			return stack.func_77952_i();
		for(int dye=0;dye<dyeNames.length;dye++)
			if(compareToOreName(stack,"dye"+dyeNames[dye]))
				return dye;
		return -1;
	}
	public static boolean isDye(ItemStack stack)
	{
		if(stack.func_190926_b())
			return false;
		if(stack.func_77973_b().equals(Items.field_151100_aR))
			return true;
		for(int dye=0;dye<dyeNames.length;dye++)
			if(compareToOreName(stack,"dye"+dyeNames[dye]))
				return true;
		return false;
	}

	public static FluidStack copyFluidStackWithAmount(FluidStack stack, int amount, boolean stripPressure)
	{
		if(stack==null)
			return null;
		FluidStack fs = new FluidStack(stack, amount);
		if(stripPressure && fs.tag!=null && fs.tag.func_74764_b("pressurized"))
		{
			fs.tag.func_82580_o("pressurized");
			if(fs.tag.func_82582_d())
				fs.tag = null;
		}
		return fs;
	}

	static long UUIDBase = 109406000905L;
	static long UUIDAdd = 01L;
	public static UUID generateNewUUID()
	{
		UUID uuid = new UUID(UUIDBase,UUIDAdd);
		UUIDAdd++;
		return uuid;
	}

	public static BlockPos toCC(Object object)
	{
		return ApiUtils.toBlockPos(object);
	}

	public static DirectionalBlockPos toDirCC(Object object, EnumFacing direction)
	{
		if(object instanceof BlockPos)
			return new DirectionalBlockPos((BlockPos)object, direction);
		if(object instanceof TileEntity)
			return new DirectionalBlockPos(((TileEntity)object).func_174877_v(), direction);
		return null;
	}

	public static boolean isBlockAt(World world, BlockPos pos, Block b, int meta)
	{
		return blockstateMatches(world.func_180495_p(pos), b, meta);
	}
	public static boolean blockstateMatches(IBlockState state, Block b, int meta)
	{
		if(state.func_177230_c().equals(b))
			return meta<0||meta==OreDictionary.WILDCARD_VALUE || state.func_177230_c().func_176201_c(state)==meta;
		return false;
	}
	public static boolean isOreBlockAt(World world, BlockPos pos, String oreName)
	{
		IBlockState state = world.func_180495_p(pos);
		ItemStack stack = new ItemStack(state.func_177230_c(),1,state.func_177230_c().func_176201_c(state));
		return compareToOreName(stack, oreName);
	}

	public static boolean canFenceConnectTo(IBlockAccess world, BlockPos pos, EnumFacing facing, Material blockMaterial)
	{
		BlockPos other = pos.func_177972_a(facing);
		IBlockState state = world.func_180495_p(other);
		Block block = world.func_180495_p(other).func_177230_c();
		if(block.canBeConnectedTo(world, other, facing.func_176734_d()))
			return true;
		BlockFaceShape blockfaceshape = state.func_193401_d(world, other, facing.func_176734_d());
		boolean flag = blockfaceshape == BlockFaceShape.MIDDLE_POLE && (state.func_185904_a()==blockMaterial || block instanceof BlockFenceGate);
		return !isExceptBlockForAttachWithFence(block) && blockfaceshape == BlockFaceShape.SOLID || flag;
	}

	private static boolean isExceptionBlockForAttaching(Block block)
	{
		return block instanceof BlockShulkerBox|| block instanceof BlockLeaves|| block instanceof BlockTrapDoor || block == Blocks.field_150461_bJ || block == Blocks.field_150383_bp || block == Blocks.field_150359_w || block == Blocks.field_150426_aN || block == Blocks.field_150432_aD || block == Blocks.field_180398_cJ || block == Blocks.field_150399_cn;
	}
	private static boolean isExceptBlockForAttachWithPiston(Block block)
	{
		return isExceptionBlockForAttaching(block) || block == Blocks.field_150331_J || block == Blocks.field_150320_F || block == Blocks.field_150332_K;
	}
	private static boolean isExceptBlockForAttachWithFence(Block block)
	{
		return isExceptBlockForAttachWithPiston(block) || block == Blocks.field_180401_cv || block == Blocks.field_150440_ba || block == Blocks.field_150423_aK || block == Blocks.field_150428_aP;
	}


	public static String formatDouble(double d, String s)
	{
		DecimalFormat df = new DecimalFormat(s);
		return df.format(d);
	}
	public static String toScientificNotation(int value, String decimalPrecision, int useKilo)
	{
		float formatted = value>=1000000000?value/1000000000f : value>=1000000?value/1000000f: value>=useKilo?value/1000f: value;
		String notation = value>=1000000000?"G" : value>=1000000?"M": value>=useKilo?"K": "";
		return formatDouble(formatted, "0."+decimalPrecision)+notation;
	}
	public static String toCamelCase(String s)
	{
		return s.substring(0,1).toUpperCase(Locale.ENGLISH) + s.substring(1).toLowerCase(Locale.ENGLISH);
	}
	static Method m_getHarvestLevel = null;
	public static String getHarvestLevelName(int lvl)
	{
		if(Loader.isModLoaded("TConstruct"))
		{
			try{
				if(m_getHarvestLevel==null)
				{
					Class clazz = Class.forName("tconstruct.library.util");
					if(clazz!=null)
						m_getHarvestLevel = clazz.getDeclaredMethod("getHarvestLevelName", int.class);
				}
				if(m_getHarvestLevel!=null)
					return (String)m_getHarvestLevel.invoke(null, lvl);
			}catch(Exception e){}
		}
		return I18n.func_135052_a(Lib.DESC_INFO+"mininglvl."+Math.max(-1, Math.min(lvl, 6)));
	}

	public static String getModVersion(String modid)
	{
		for(ModContainer container : Loader.instance().getActiveModList())
			if(container.getModId().equalsIgnoreCase(modid))
				return container.getVersion();
		return "";
	}

	public static boolean tilePositionMatch(TileEntity tile0, TileEntity tile1)
	{
		return tile0.func_174877_v().equals(tile1.func_174877_v());
	}

	public static EnumFacing rotateFacingTowardsDir(EnumFacing f, EnumFacing dir)
	{
		if(dir==EnumFacing.NORTH)
			return f;
		else if(dir==EnumFacing.SOUTH && f.func_176740_k()!=Axis.Y)
			return f.func_176746_e().func_176746_e();
		else if(dir==EnumFacing.WEST && f.func_176740_k()!=Axis.Y)
			return f.func_176735_f();
		else if(dir==EnumFacing.EAST && f.func_176740_k()!=Axis.Y)
			return f.func_176746_e();
		else if(dir==EnumFacing.DOWN && f.func_176740_k()!=Axis.Y)
			return f.func_176732_a(Axis.X);
		else if(dir==EnumFacing.UP && f.func_176740_k()!=Axis.X)
			return f.func_176732_a(Axis.X).func_176734_d();
		return f;
	}

	public static RayTraceResult getMovingObjectPositionFromPlayer(World world, EntityLivingBase living, boolean bool)
	{
		float f = 1.0F;
		float f1 = living.field_70127_C + (living.field_70125_A - living.field_70127_C) * f;
		float f2 = living.field_70126_B + (living.field_70177_z - living.field_70126_B) * f;
		double d0 = living.field_70169_q + (living.field_70165_t - living.field_70169_q) * (double)f;
		double d1 = living.field_70167_r + (living.field_70163_u - living.field_70167_r) * (double)f + (double)(world.field_72995_K ? living.func_70047_e() - (living instanceof EntityPlayer?((EntityPlayer)living).getDefaultEyeHeight():0) : living.func_70047_e()); // isRemote check to revert changes to ray trace position due to adding the eye height clientside and player yOffset differences
		double d2 = living.field_70166_s + (living.field_70161_v - living.field_70166_s) * (double)f;
		Vec3d vec3 = new Vec3d(d0, d1, d2);
		float f3 = MathHelper.func_76134_b(-f2 * 0.017453292F - (float)Math.PI);
		float f4 = MathHelper.func_76126_a(-f2 * 0.017453292F - (float)Math.PI);
		float f5 = -MathHelper.func_76134_b(-f1 * 0.017453292F);
		float f6 = MathHelper.func_76126_a(-f1 * 0.017453292F);
		float f7 = f4 * f5;
		float f8 = f3 * f5;
		double d3 = 5.0D;
		if (living instanceof EntityPlayerMP)
			d3 = ((EntityPlayerMP)living).field_71134_c.getBlockReachDistance();

		Vec3d vec31 = vec3.func_72441_c((double)f7 * d3, (double)f6 * d3, (double)f8 * d3);
		return world.func_147447_a(vec3, vec31, bool, !bool, false);
	}
	public static boolean canBlocksSeeOther(World world, BlockPos cc0, BlockPos cc1, Vec3d pos0, Vec3d pos1)
	{
		HashSet<BlockPos> inter = rayTrace(pos0, pos1, world);
		Iterator<BlockPos> it = inter.iterator();
		while (it.hasNext()) {
			BlockPos cc = it.next();
			if (!cc.equals(cc0)&&!cc.equals(cc1))
				return false;
		}
		return true;
	}

	public static List<EntityLivingBase> getTargetsInCone(World world, Vec3d start, Vec3d dir, float spreadAngle, float truncationLength)
	{
		double length = dir.func_72433_c();
		Vec3d dirNorm = dir.func_72432_b();
		double radius = Math.tan(spreadAngle/2)*length;

		Vec3d endLow = start.func_178787_e(dir).func_178786_a(radius,radius,radius);
		Vec3d endHigh = start.func_178787_e(dir).func_72441_c(radius,radius,radius);

		AxisAlignedBB box = new AxisAlignedBB(minInArray(start.field_72450_a,endLow.field_72450_a,endHigh.field_72450_a), minInArray(start.field_72448_b,endLow.field_72448_b,endHigh.field_72448_b), minInArray(start.field_72449_c,endLow.field_72449_c,endHigh.field_72449_c),
				maxInArray(start.field_72450_a,endLow.field_72450_a,endHigh.field_72450_a), maxInArray(start.field_72448_b,endLow.field_72448_b,endHigh.field_72448_b), maxInArray(start.field_72449_c,endLow.field_72449_c,endHigh.field_72449_c));

		List<EntityLivingBase> list = world.func_72872_a(EntityLivingBase.class, box);
		Iterator<EntityLivingBase> iterator = list.iterator();
		while(iterator.hasNext())
		{
			EntityLivingBase e = iterator.next();
			if(!isPointInCone(start, dirNorm, radius, length, truncationLength, e.func_174791_d().func_178788_d(start)))
				iterator.remove();
		}
		return list;
	}

	public static boolean isPointInConeByAngle(Vec3d start, Vec3d normDirection, double aperture, double length, Vec3d relativePoint)
	{
		return isPointInCone(start, normDirection, Math.tan(aperture/2)*length, length, 0, relativePoint);
	}
	public static boolean isPointInCone(Vec3d start, Vec3d normDirection, double radius, double length, Vec3d relativePoint)
	{
		return isPointInCone(start, normDirection, radius, length, 0, relativePoint);
	}
	public static boolean isPointInConeByAngle(Vec3d start, Vec3d normDirection, float aperture, double length, float truncationLength, Vec3d relativePoint)
	{
		return isPointInCone(start, normDirection, Math.tan(aperture/2)*length, length, truncationLength, relativePoint);
	}
	/**
	 * Checks if  point is contained within a cone in 3D space
	 *
	 * @param start tip of the cone
	 * @param normDirection normalized (length==1) vector, direction of cone
	 * @param radius radius at the end of the cone
	 * @param length length of the cone
	 * @param truncationLength optional lenght at which the cone is truncated (flat tip)
	 * @param relativePoint point to be checked, relative to {@code start}
	 */
	public static boolean isPointInCone(Vec3d start, Vec3d normDirection, double radius, double length, float truncationLength, Vec3d relativePoint)
	{
		double projectedDist = relativePoint.func_72430_b(normDirection); //Orthogonal projection, establishing point's distance on cone direction vector
		if(projectedDist<truncationLength || projectedDist>length) //If projected distance is before truncation or beyond length, point not contained
			return false;

		double radiusAtDist = projectedDist/length * radius; //Radius of the cone at the projected distance
		Vec3d orthVec = relativePoint.func_178788_d(normDirection.func_186678_a(projectedDist)); //Orthogonal vector between point and cone direction

		return orthVec.func_189985_c()<(radiusAtDist*radiusAtDist); //Check if Vector's length is shorter than radius -> point in cone
	}
	public static boolean isPointInTriangle(Vec3d tA, Vec3d tB, Vec3d tC, Vec3d point)
	{
		//Distance vectors to A (focuspoint of triangle)
		Vec3d v0 = tC.func_178788_d(tA);
		Vec3d v1 = tB.func_178788_d(tA);
		Vec3d v2 = point.func_178788_d(tA);

		return isPointInTriangle(v0, v1, v2);
	}
	private static boolean isPointInTriangle(Vec3d leg0, Vec3d leg1, Vec3d targetVec)
	{
		//Dot products
		double dot00 = leg0.func_72430_b(leg0);
		double dot01 = leg0.func_72430_b(leg1);
		double dot02 = leg0.func_72430_b(targetVec);
		double dot11 = leg1.func_72430_b(leg1);
		double dot12 = leg1.func_72430_b(targetVec);

		//Barycentric coordinates
		double invDenom = 1 / (dot00 * dot11 - dot01 * dot01);
		double u = (dot11 * dot02 - dot01 * dot12) * invDenom;
		double v = (dot00 * dot12 - dot01 * dot02) * invDenom;

		return (u >= 0) && (v >= 0) && (u + v < 1);
	}
	private static Vec3d getVectorForRotation(float pitch, float yaw)
	{
		float f = MathHelper.func_76134_b(-yaw * 0.017453292F - (float)Math.PI);
		float f1 = MathHelper.func_76126_a(-yaw * 0.017453292F - (float)Math.PI);
		float f2 = -MathHelper.func_76134_b(-pitch * 0.017453292F);
		float f3 = MathHelper.func_76126_a(-pitch * 0.017453292F);
		return new Vec3d((double)(f1 * f2), (double)f3, (double)(f * f2));
	}

	public static boolean isHammer(ItemStack stack)
	{
		if(stack.func_190926_b())
			return false;
		return stack.func_77973_b().getToolClasses(stack).contains(Lib.TOOL_HAMMER);
	}

	public static boolean canBlockDamageSource(EntityLivingBase entity, DamageSource damageSourceIn)
	{
		if(!damageSourceIn.func_76363_c() && entity.func_184585_cz())
		{
			Vec3d vec3d = damageSourceIn.func_188404_v();
			if(vec3d != null)
			{
				Vec3d vec3d1 = entity.func_70676_i(1.0F);
				Vec3d vec3d2 = vec3d.func_72444_a(entity.func_174791_d()).func_72432_b();
				vec3d2 = new Vec3d(vec3d2.field_72450_a, 0.0D, vec3d2.field_72449_c);
				return vec3d2.func_72430_b(vec3d1) < 0;
			}
		}
		return false;
	}

	public static Vec3d getFlowVector(World world, BlockPos pos)
	{
		IBlockState state = world.func_180495_p(pos);
		if(state.func_177230_c() instanceof BlockFluidBase)
			return ((BlockFluidBase)state.func_177230_c()).getFlowVector(world, pos);
		else if( !(state.func_177230_c() instanceof BlockLiquid))
			return new Vec3d(0, 0, 0);

		BlockLiquid block = (BlockLiquid)state.func_177230_c();
		Vec3d vec3 = new Vec3d(0.0D, 0.0D, 0.0D);
		Material mat = state.func_185904_a();
		int i = getEffectiveFlowDecay(world, pos, mat);

		for(EnumFacing enumfacing : EnumFacing.Plane.HORIZONTAL)
		{
			BlockPos blockpos = pos.func_177972_a(enumfacing);
			int j = getEffectiveFlowDecay(world, blockpos, mat);
			if(j<0)
			{
				if(!world.func_180495_p(blockpos).func_185904_a().func_76230_c())
				{
					j = getEffectiveFlowDecay(world, blockpos.func_177977_b(), mat);
					if(j>=0)
					{
						int k = j - (i - 8);
						vec3 = vec3.func_72441_c((double)((blockpos.func_177958_n() - pos.func_177958_n()) * k), (double)((blockpos.func_177956_o() - pos.func_177956_o()) * k), (double)((blockpos.func_177952_p() - pos.func_177952_p()) * k));
					}
				}
			}
			else if(j>=0)
			{
				int l = j - i;
				vec3 = vec3.func_72441_c((double)((blockpos.func_177958_n() - pos.func_177958_n()) * l), (double)((blockpos.func_177956_o() - pos.func_177956_o()) * l), (double)((blockpos.func_177952_p() - pos.func_177952_p()) * l));
			}
		}

		if(state.func_177229_b(BlockLiquid.field_176367_b).intValue()>=8)
		{
			for(EnumFacing enumfacing1 : EnumFacing.Plane.HORIZONTAL)
			{
				BlockPos blockpos1 = pos.func_177972_a(enumfacing1);
				if(block.func_176212_b(world, blockpos1, enumfacing1) || block.func_176212_b(world, blockpos1.func_177984_a(), enumfacing1))
				{
					vec3 = vec3.func_72432_b().func_72441_c(0.0D, -6.0D, 0.0D);
					break;
				}
			}
		}
		return vec3.func_72432_b();
	}
	static int getEffectiveFlowDecay(IBlockAccess world, BlockPos pos, Material mat)
	{
		IBlockState state = world.func_180495_p(pos);
		if(state.func_185904_a() != mat)
			return -1;
		int l = state.func_177230_c().func_176201_c(state);
		if (l >= 8)
			l = 0;
		return l;
	}
	public static Vec3d addVectors(Vec3d vec0, Vec3d vec1)
	{
		return vec0.func_72441_c(vec1.field_72450_a,vec1.field_72448_b,vec1.field_72449_c);
	}

	public static double minInArray(double... f)
	{
		if(f.length<1)
			return 0;
		double min = f[0];
		for(int i=1; i<f.length; i++)
			min = Math.min(min, f[i]);
		return min;
	}
	public static double maxInArray(double... f)
	{
		if(f.length<1)
			return 0;
		double max = f[0];
		for(int i=1; i<f.length; i++)
			max = Math.max(max, f[i]);
		return max;
	}

	public static boolean isVecInEntityHead(EntityLivingBase entity, Vec3d vec)
	{
		if(entity.field_70131_O/entity.field_70130_N<2)//Crude check to see if the entity is bipedal or at least upright (this should work for blazes)
			return false;
		double d = vec.field_72448_b-(entity.field_70163_u+entity.func_70047_e());
		return Math.abs(d) < .25;
	}

	public static void unlockIEAdvancement(EntityPlayer player, String name)
	{
		if(player instanceof EntityPlayerMP)
		{
			PlayerAdvancements advancements = ((EntityPlayerMP)player).func_192039_O();
			AdvancementManager manager = ((WorldServer)player.func_130014_f_()).func_191952_z();
			Advancement advancement = manager.func_192778_a(new ResourceLocation(ImmersiveEngineering.MODID, name));
			if(advancement!=null)
				advancements.func_192750_a(advancement, "code_trigger");
		}
	}

	public static NBTTagCompound getRandomFireworkExplosion(Random rand, int preType)
	{
		NBTTagCompound tag = new NBTTagCompound();
		NBTTagCompound expl = new NBTTagCompound();
		expl.func_74757_a("Flicker", true);
		expl.func_74757_a("Trail", true);
		int[] colors = new int[rand.nextInt(8) + 1];
		for (int i = 0; i < colors.length; i++)
		{
			int j = rand.nextInt(11)+1;
			if(j>2)
				j++;
			if(j>6)
				j+=2;
			//no black, brown, light grey, grey or white
			colors[i] = ItemDye.field_150922_c[j];
		}
		expl.func_74783_a("Colors", colors);
		int type = preType>=0?preType: rand.nextInt(4);
		if(preType<0 && type==3)
			type = 4;
		expl.func_74774_a("Type", (byte) type);
		NBTTagList list = new NBTTagList();
		list.func_74742_a(expl);
		tag.func_74782_a("Explosions", list);

		return tag;
	}

	public static FluidStack drainFluidBlock(World world, BlockPos pos, boolean doDrain)
	{
		Block b = world.func_180495_p(pos).func_177230_c();
		Fluid f = FluidRegistry.lookupFluidForBlock(b);

		if(f!=null)
		{
			if(b instanceof IFluidBlock)
			{
				if(((IFluidBlock)b).canDrain(world, pos))
					return ((IFluidBlock) b).drain(world, pos, doDrain);
				else
					return null;
			}
			else
			{
				if(b.func_176201_c(world.func_180495_p(pos))==0)
				{
					if(doDrain)
						world.func_175698_g(pos);
					return new FluidStack(f, 1000);
				}
				return null;
			}
		}
		return null;
	}

	public static Fluid getRelatedFluid(World w, BlockPos pos)
	{
		Block b = w.func_180495_p(pos).func_177230_c();
		return FluidRegistry.lookupFluidForBlock(b);
	}

	public static boolean placeFluidBlock(World world, BlockPos pos, FluidStack fluid)
	{
		if(fluid==null || fluid.getFluid()==null)
			return false;
		IBlockState state = world.func_180495_p(pos);
		Block b = state.func_177230_c();
		Block fluidBlock = fluid.getFluid().getBlock();

		if(Blocks.field_150355_j.equals(fluidBlock))
			fluidBlock = Blocks.field_150358_i;
		else if(Blocks.field_150353_l.equals(fluidBlock))
			fluidBlock = Blocks.field_150356_k;

		boolean canPlace = b==null||b.isAir(state,world,pos)||b.func_176200_f(world,pos);

		if(fluidBlock!=null && canPlace && fluid.amount>=1000)
		{
			boolean placed = false;
			if ((fluidBlock instanceof BlockFluidBase))
			{
				BlockFluidBase blockFluid = (BlockFluidBase)fluidBlock;
				placed = world.func_175656_a(pos, fluidBlock.func_176203_a(blockFluid.getMaxRenderHeightMeta()));
			}
			else
				placed = world.func_175656_a(pos, fluidBlock.func_176223_P());
			if(placed)
				fluid.amount -= 1000;
			return placed;
		}
		return false;
	}

//	public static Collection<ItemStack> getContainersFilledWith(FluidStack fluidStack)
//	{
//		List<ItemStack> containers = new ArrayList();
//		for (FluidContainerRegistry.FluidContainerData data : FluidContainerRegistry.getRegisteredFluidContainerData())
//			if(data.fluid.containsFluid(fluidStack))
//				containers.add(data.filledContainer);
//		return containers;
//	}

	//	public static String nameFromStack(ItemStack stack)
	//	{
	//		if(stack==null)
	//			return "";
	//		try
	//		{
	//			return GameData.getItemRegistry().getNameForObject(stack.getItem());
	//		}
	//		catch (NullPointerException e) {}
	//		return "";
	//	}

	public static IBlockState getStateFromItemStack(ItemStack stack)
	{
		if(stack.func_190926_b())
			return null;
		Block block = getBlockFromItem(stack.func_77973_b());
		if(block!=null)
			return block.func_176203_a(stack.func_77952_i());
		return null;
	}
	public static Block getBlockFromItem(Item item)
	{
		if(item==Items.field_151066_bu)
			return Blocks.field_150383_bp;
		return Block.func_149634_a(item);
	}

	public static boolean canInsertStackIntoInventory(TileEntity inventory, ItemStack stack, EnumFacing side)
	{
		if(!stack.func_190926_b() && inventory!=null && inventory.hasCapability(CapabilityItemHandler.ITEM_HANDLER_CAPABILITY, side))
		{
			IItemHandler handler = inventory.getCapability(CapabilityItemHandler.ITEM_HANDLER_CAPABILITY, side);
			ItemStack temp = ItemHandlerHelper.insertItem(handler, stack.func_77946_l(), true);
			return temp.func_190926_b()||temp.func_190916_E() < stack.func_190916_E();
		}
		return false;
	}
	public static ItemStack insertStackIntoInventory(TileEntity inventory, ItemStack stack, EnumFacing side)
	{
		if(!stack.func_190926_b() && inventory!=null && inventory.hasCapability(CapabilityItemHandler.ITEM_HANDLER_CAPABILITY, side))
		{
			IItemHandler handler = inventory.getCapability(CapabilityItemHandler.ITEM_HANDLER_CAPABILITY, side);
			ItemStack temp = ItemHandlerHelper.insertItem(handler, stack.func_77946_l(), true);
			if(temp.func_190926_b() || temp.func_190916_E() < stack.func_190916_E())
				return ItemHandlerHelper.insertItem(handler, stack, false);
		}
		return stack;
	}
	public static ItemStack insertStackIntoInventory(TileEntity inventory, ItemStack stack, EnumFacing side, boolean simulate)
	{
		if(inventory!=null && !stack.func_190926_b() && inventory.hasCapability(CapabilityItemHandler.ITEM_HANDLER_CAPABILITY, side))
		{
			IItemHandler handler = inventory.getCapability(CapabilityItemHandler.ITEM_HANDLER_CAPABILITY, side);
			return ItemHandlerHelper.insertItem(handler, stack.func_77946_l(), simulate);
		}
		return stack;
	}
	public static void dropStackAtPos(World world, BlockPos pos, ItemStack stack, EnumFacing facing)
	{
		if(!stack.func_190926_b())
		{
			EntityItem ei = new EntityItem(world, pos.func_177958_n()+.5,pos.func_177956_o()+.5,pos.func_177952_p()+.5, stack.func_77946_l());
			ei.field_70181_x = 0.025000000372529D;
			if(facing!=null)
			{
				ei.field_70159_w = (0.075F * facing.func_82601_c());
				ei.field_70179_y = (0.075F * facing.func_82599_e());
			}
			world.func_72838_d(ei);
		}
	}
	public static void dropStackAtPos(World world, BlockPos pos, ItemStack stack)
	{
		dropStackAtPos(world, pos, stack, null);
	}
	//	public static ItemStack insertStackIntoInventory(IInventory inventory, ItemStack stack, EnumFacing side)
	//	{
	//		if (stack == null || inventory == null)
	//			return null;
	//		int stackSize = stack.stackSize;
	//		if (inventory instanceof ISidedInventory)
	//		{
	//			ISidedInventory sidedInv = (ISidedInventory) inventory;
	//			int slots[] = sidedInv.getSlotsForFace(side);
	//			if (slots == null)
	//				return stack;
	//			for (int i=0; i<slots.length && stack!=null; i++)
	//			{
	//				if (sidedInv.canInsertItem(slots[i], stack, side))
	//				{
	//					ItemStack existingStack = inventory.getStackInSlot(slots[i]);
	//					if(OreDictionary.itemMatches(existingStack, stack, true)&&Utils.compareItemNBT(stack, existingStack))
	//						stack = addToOccupiedSlot(sidedInv, slots[i], stack, existingStack);
	//				}
	//			}
	//			for (int i=0; i<slots.length && stack!=null; i++)
	//				if (inventory.getStackInSlot(slots[i]) == null && sidedInv.canInsertItem(slots[i], stack, side))
	//					stack = addToEmptyInventorySlot(sidedInv, slots[i], stack);
	//		}
	//		else
	//		{
	//			int invSize = inventory.getSizeInventory();
	//			for (int i=0; i<invSize && stack!=null; i++)
	//			{
	//				ItemStack existingStack = inventory.getStackInSlot(i);
	//				if (OreDictionary.itemMatches(existingStack, stack, true)&&Utils.compareItemNBT(stack, existingStack))
	//					stack = addToOccupiedSlot(inventory, i, stack, existingStack);
	//			}
	//			for (int i=0; i<invSize && stack!=null; i++)
	//				if (inventory.getStackInSlot(i) == null)
	//					stack = addToEmptyInventorySlot(inventory, i, stack);
	//		}
	//		if (stack == null || stack.stackSize != stackSize)
	//			inventory.markDirty();
	//		return stack;
	//	}

	public static ItemStack addToEmptyInventorySlot(IInventory inventory, int slot, ItemStack stack)
	{
		if (!inventory.func_94041_b(slot, stack)) {
			return stack;
		}
		int stackLimit = inventory.func_70297_j_();
		inventory.func_70299_a(slot, copyStackWithAmount(stack, Math.min(stack.func_190916_E(), stackLimit)));
		return stackLimit >= stack.func_190916_E() ? ItemStack.field_190927_a : stack.func_77979_a(stack.func_190916_E() - stackLimit);
	}
	public static ItemStack addToOccupiedSlot(IInventory inventory, int slot, ItemStack stack, ItemStack existingStack)
	{
		int stackLimit = Math.min(inventory.func_70297_j_(), stack.func_77976_d());
		if (stack.func_190916_E() + existingStack.func_190916_E() > stackLimit) {
			int stackDiff = stackLimit - existingStack.func_190916_E();
			existingStack.func_190920_e(stackLimit);
			stack.func_190918_g(stackDiff);
			inventory.func_70299_a(slot, existingStack);
			return stack;
		}
		existingStack.func_190917_f(min(stack.func_190916_E(), stackLimit));
		inventory.func_70299_a(slot, existingStack);
		return stackLimit >= stack.func_190916_E() ? ItemStack.field_190927_a : stack.func_77979_a(stack.func_190916_E() - stackLimit);
	}


	//	public static boolean canInsertStackIntoInventory(IInventory inventory, ItemStack stack, EnumFacing side)
	//	{
	//		if(stack == null || inventory == null)
	//			return false;
	//		if(inventory instanceof ISidedInventory)
	//		{
	//			ISidedInventory sidedInv = (ISidedInventory) inventory;
	//			int slots[] = sidedInv.getSlotsForFace(side);
	//			if(slots == null)
	//				return false;
	//			for(int i=0; i<slots.length && stack!=null; i++)
	//			{
	//				if(sidedInv.canInsertItem(slots[i], stack, side) && sidedInv.isItemValidForSlot(slots[i], stack))
	//				{
	//					ItemStack existingStack = inventory.getStackInSlot(slots[i]);
	//					if(existingStack==null)
	//						return true;
	//					else
	//						if(OreDictionary.itemMatches(existingStack, stack, true)&&Utils.compareItemNBT(stack, existingStack))
	//							if(existingStack.stackSize+stack.stackSize<inventory.getInventoryStackLimit() && existingStack.stackSize+stack.stackSize<existingStack.getMaxStackSize())
	//								return true;
	//				}
	//			}
	//		}
	//		else
	//		{
	//			int invSize = inventory.getSizeInventory();
	//			for(int i=0; i<invSize && stack!=null; i++)
	//				if(inventory.isItemValidForSlot(i, stack))
	//				{
	//					ItemStack existingStack = inventory.getStackInSlot(i);
	//					if(existingStack==null)
	//						return true;
	//					else
	//						if(OreDictionary.itemMatches(existingStack, stack, true)&&Utils.compareItemNBT(stack, existingStack))
	//							if(existingStack.stackSize+stack.stackSize<inventory.getInventoryStackLimit() && existingStack.stackSize+stack.stackSize<existingStack.getMaxStackSize())
	//								return true;
	//				}
	//		}
	//		return false;
	//	}

	public static ItemStack fillFluidContainer(IFluidHandler handler, ItemStack containerIn, ItemStack containerOut, @Nullable EntityPlayer player)
	{
		if(containerIn==null || containerIn.func_190926_b())
			return ItemStack.field_190927_a;
		if(containerIn.func_77942_o() && containerIn.func_77978_p().func_82582_d())
			containerIn.func_77982_d(null);

		FluidActionResult result = FluidUtil.tryFillContainer(containerIn, handler, Integer.MAX_VALUE, player, false);
		if(result.isSuccess())
		{
			final ItemStack full = result.getResult();
			if((containerOut.func_190926_b() || OreDictionary.itemMatches(containerOut, full,true)))
			{
				if (!containerOut.func_190926_b() && containerOut.func_190916_E() + full.func_190916_E() > containerOut.func_77976_d())
					return ItemStack.field_190927_a;
				result = FluidUtil.tryFillContainer(containerIn, handler, Integer.MAX_VALUE, player, true);
				if (result.isSuccess()) {
					return result.getResult();
				}
			}
		}
		return ItemStack.field_190927_a;
	}
	public static ItemStack drainFluidContainer(IFluidHandler handler, ItemStack containerIn, ItemStack containerOut, @Nullable EntityPlayer player)
	{
		if(containerIn==null || containerIn.func_190926_b())
			return ItemStack.field_190927_a;

		if(containerIn.func_77942_o() && containerIn.func_77978_p().func_82582_d())
			containerIn.func_77982_d(null);

		FluidActionResult result = FluidUtil.tryEmptyContainer(containerIn, handler, Integer.MAX_VALUE, player, false);
		if(result.isSuccess())
		{
			ItemStack empty = result.getResult();
			if ((containerOut.func_190926_b() || OreDictionary.itemMatches(containerOut,empty,true)))
			{
				if (!containerOut.func_190926_b() && containerOut.func_190916_E() + empty.func_190916_E() > containerOut.func_77976_d())
					return ItemStack.field_190927_a;
				result = FluidUtil.tryEmptyContainer(containerIn, handler, Integer.MAX_VALUE, player, true);
				if (result.isSuccess()) {
					return result.getResult();
				}
			}
		}
		return ItemStack.field_190927_a;

	}

//	public static FluidStack getFluidFromItemStack(ItemStack stack)
//	{
//		if(stack==null)
//			return null;
//		FluidStack fluid = FluidContainerRegistry.getFluidForFilledItem(stack);
//		if(fluid != null)
//			return fluid;
//		else if(stack.getItem() instanceof IFluidContainerItem)
//			return ((IFluidContainerItem)stack.getItem()).getFluid(stack);
//		return null;
//	}

	public static boolean isFluidRelatedItemStack(ItemStack stack)
	{
		if(stack.func_190926_b())
			return false;
		return stack.hasCapability(CapabilityFluidHandler.FLUID_HANDLER_ITEM_CAPABILITY, null);
	}

	public static IRecipe findRecipe(InventoryCrafting crafting, World world)
	{
		return CraftingManager.func_192413_b(crafting, world);
	}

	public static NonNullList<ItemStack> createNonNullItemStackListFromArray(ItemStack[] stacks)
	{
		NonNullList<ItemStack> list = NonNullList.func_191197_a(stacks.length, ItemStack.field_190927_a);
		for (int i = 0; i < stacks.length; i++)
		{
			list.set(i, stacks[i]);
		}
		return list;
	}

	public static NonNullList<ItemStack> createNonNullItemStackListFromItemStack(ItemStack stack)
	{
		NonNullList<ItemStack> list = NonNullList.func_191197_a(1, ItemStack.field_190927_a);
		list.set(0, stack);
		return list;
	}

	public static float[] rotateToFacing(float[] in, EnumFacing facing)
	{
		for (int i = 0; i < in.length; i++)
			in[i] -= .5F;
		float[] ret = new float[in.length];
		for (int i = 0; i < in.length; i += 3)
			for (int j = 0; j < 3; j++)
			{
				if (j == 0)
					ret[i + j] = in[i + 0] * facing.func_82599_e() +
							in[i + 1] * facing.func_82601_c() +
							in[i + 2] * facing.func_96559_d();
				else if (j == 1)
					ret[i + j] = in[i + 0] * facing.func_82601_c() +
							in[i + 1] * facing.func_96559_d() +
							in[i + 2] * facing.func_82599_e();
				else
					ret[i + j] = in[i + 0] * facing.func_96559_d() +
							in[i + 1] * facing.func_82599_e() +
							in[i + 2] * facing.func_82601_c();
			}
		for (int i = 0; i < in.length; i++)
			ret[i] += .5;
		return ret;
	}

	public static int hashBlockstate(IBlockState state, Set<Object> ignoredProperties, boolean includeExtended)
	{
		int val = 0;
		final int prime = 31;
		for (IProperty<?> n : state.func_177227_a())
			if (!ignoredProperties.contains(n))
			{
				Object o = state.func_177229_b(n);
				val = prime * val + (o == null ? 0 : o.hashCode());
			}
		if (includeExtended&&state instanceof IExtendedBlockState)
		{
			IExtendedBlockState ext = (IExtendedBlockState) state;
			for (IUnlistedProperty<?> n : ext.getUnlistedNames())
				if (!ignoredProperties.contains(n))
				{
					Object o = ext.getValue(n);
					val = prime * val + (o == null ? 0 : o.hashCode());
				}
		}
		return val;
	}

	public static boolean areStatesEqual(IBlockState state, IBlockState other, Set<Object> ignoredProperties, boolean includeExtended)
	{
		for(IProperty<?> i : state.func_177227_a())
		{
			if(!other.func_177228_b().containsKey(i))
				return false;
			if (ignoredProperties.contains(i))
				continue;
			Object valThis = state.func_177229_b(i);
			Object valOther = other.func_177229_b(i);
			if(valThis==null&&valOther==null)
				continue;
			else if(valOther == null || !valOther.equals(state.func_177229_b(i)))
				return false;
		}
		if (includeExtended)
		{
			if (state instanceof IExtendedBlockState^other instanceof IExtendedBlockState)
				return false;
			if (state instanceof IExtendedBlockState)
			{
				IExtendedBlockState extState = (IExtendedBlockState) state;
				IExtendedBlockState extOther = (IExtendedBlockState) other;
				for (IUnlistedProperty<?> i : extState.getUnlistedNames())
				{
					if (!extOther.getUnlistedProperties().containsKey(i))
						return false;
					if (ignoredProperties.contains(i))
						continue;
					Object valThis = extState.getValue(i);
					Object valOther = extOther.getValue(i);
					if (valThis == null && valOther == null)
						continue;
					else if (valOther == null || !valOther.equals(valThis))
						return false;
				}
			}
		}
		return true;
	}

	public static boolean areArraysEqualIncludingBlockstates(Object[] a, Object[] a2)
	{
		if (a == a2)
			return true;
		if (a == null || a2 == null)
			return false;

		int length = a.length;
		if (a2.length != length)
			return false;

		for (int i = 0; i < length; i++)
		{
			Object o1 = a[i];
			Object o2 = a2[i];
			if (o1 instanceof IBlockState && o2 instanceof IBlockState)
			{
				if (!areStatesEqual((IBlockState)o1, (IBlockState) o2, ImmutableSet.of(), false))
					return false;
			}
			else if (!(o1 == null ? o2 == null : o1.equals(o2)))
				return false;
		}
		return true;
	}

	public static class InventoryCraftingFalse extends InventoryCrafting
	{
		private static final Container nullContainer = new Container()
		{
			@Override
			public void func_75130_a(IInventory paramIInventory){}
			@Override
			public boolean func_75145_c(EntityPlayer p_75145_1_)
			{
				return false;
			}
		};
		public InventoryCraftingFalse(int w, int h)
		{
			super(nullContainer, w, h);
		}

		public static InventoryCrafting createFilledCraftingInventory(int w, int h, NonNullList<ItemStack> stacks)
		{
			InventoryCrafting invC = new Utils.InventoryCraftingFalse(w, h);
			for(int j = 0; j < w * h; j++)
				if(!stacks.get(j).func_190926_b())
					invC.func_70299_a(j, stacks.get(j).func_77946_l());
			return invC;
		}
	}

	public static HashSet<BlockPos> rayTrace(Vec3d start, Vec3d end, World world)
	{
		return rayTrace(start, end, world, (p)->{});
	}
	public static HashSet<BlockPos> rayTrace(Vec3d start, Vec3d end, World world, Consumer<BlockPos> out)
	{
		HashSet<BlockPos> ret = new HashSet<BlockPos>();
		HashSet<BlockPos> checked = new HashSet<BlockPos>();
		// x
		if (start.field_72450_a>end.field_72450_a)
		{
			Vec3d tmp = start;
			start = end;
			end = tmp;
		}
		double min = start.field_72450_a;
		double dif =end.field_72450_a-min;
		double lengthAdd = Math.ceil(min)-start.field_72450_a;
		Vec3d mov = start.func_178788_d(end);
		if (mov.field_72450_a!=0)
		{
			mov = scalarProd(mov, 1 / mov.field_72450_a);
			ray(dif, mov, start, lengthAdd, ret, world, checked, out);
		}
		// y
		if (mov.field_72448_b!=0)
		{
			if (start.field_72448_b>end.field_72448_b)
			{
				Vec3d tmp = start;
				start = end;
				end = tmp;
			}
			min = start.field_72448_b;
			dif = end.field_72448_b-min;
			lengthAdd = Math.ceil(min)-start.field_72448_b;
			mov = start.func_178788_d(end);
			mov = scalarProd(mov, 1/mov.field_72448_b);

			ray(dif, mov, start, lengthAdd, ret, world, checked, out);
		}

		// z
		if (mov.field_72449_c!=0)
		{
			if (start.field_72449_c>end.field_72449_c)
			{
				Vec3d tmp = start;
				start = end;
				end = tmp;
			}
			min = start.field_72449_c;
			dif = end.field_72449_c - min;
			lengthAdd = Math.ceil(min)-start.field_72449_c;
			mov = start.func_178788_d(end);
			mov = scalarProd(mov, 1 / mov.field_72449_c);

			ray(dif, mov, start, lengthAdd, ret, world, checked, out);
		}
		return ret;
	}
	private static void ray(double dif, Vec3d mov, Vec3d start, double lengthAdd, HashSet<BlockPos> ret, World world, HashSet<BlockPos> checked, Consumer<BlockPos> out)
	{
		//Do NOT set this to true unless for debugging. Causes blocks to be placed along the traced ray
		boolean place = false;
		double standartOff = .0625;
		for (int i = 0; i < dif; i++)
		{
			Vec3d pos = addVectors(start, scalarProd(mov, i + lengthAdd+standartOff));
			Vec3d posNext = addVectors(start,
					scalarProd(mov, i + 1 + lengthAdd+standartOff));
			Vec3d posPrev = addVectors(start,
					scalarProd(mov, i + lengthAdd-standartOff));
			Vec3d posVeryPrev = addVectors(start,
					scalarProd(mov, i - 1 + lengthAdd-standartOff));

			BlockPos blockPos = new BlockPos((int) Math.floor(pos.field_72450_a),
					(int) Math.floor(pos.field_72448_b), (int) Math.floor(pos.field_72449_c));
			Block b;
			IBlockState state;
			if (!checked.contains(blockPos)&&i + lengthAdd+standartOff<dif)
			{
				state = world.func_180495_p(blockPos);
				b = state.func_177230_c();
				if (b.func_176209_a(state, false) && state.func_185910_a(world, blockPos, pos, posNext) != null)
					ret.add(blockPos);
				//				if (place)
				//					world.setBlockState(blockPos, tmp);
				checked.add(blockPos);
				out.accept(blockPos);
			}
			blockPos = new BlockPos((int) Math.floor(posPrev.field_72450_a), (int) Math.floor(posPrev.field_72448_b), (int) Math.floor(posPrev.field_72449_c));
			if (!checked.contains(blockPos)&&i + lengthAdd-standartOff<dif)
			{
				state = world.func_180495_p(blockPos);
				b = state.func_177230_c();
				if (b.func_176209_a(state, false) && state.func_185910_a(world, blockPos, posVeryPrev, posPrev) != null)
					ret.add(blockPos);
				//				if (place)
				//					world.setBlock(blockPos.posX, blockPos.posY, blockPos.posZ, tmp);
				checked.add(blockPos);
				out.accept(blockPos);
			}
		}
	}
	public static Vec3d scalarProd(Vec3d v, double s)
	{
		return new Vec3d(v.field_72450_a*s, v.field_72448_b*s, v.field_72449_c*s);
	}
	public static BlockPos rayTraceForFirst(Vec3d start, Vec3d end, World w, Set<BlockPos> ignore)
	{
		HashSet<BlockPos> trace = rayTrace(start, end, w);
		for (BlockPos cc:ignore)
			trace.remove(cc);
		if (start.field_72450_a!=end.field_72450_a)
			trace = findMinOrMax(trace, start.field_72450_a>end.field_72450_a, 0);
		if (start.field_72448_b!=end.field_72448_b)
			trace = findMinOrMax(trace, start.field_72448_b>end.field_72448_b, 0);
		if (start.field_72449_c!=end.field_72449_c)
			trace = findMinOrMax(trace, start.field_72449_c>end.field_72449_c, 0);
		if (trace.size()>0)
		{
			BlockPos ret = trace.iterator().next();
			return ret;
		}
		return null;
	}
	public static HashSet<BlockPos> findMinOrMax(HashSet<BlockPos> in, boolean max, int coord) {
		HashSet<BlockPos> ret = new HashSet<BlockPos>();
		int currMinMax = max?Integer.MIN_VALUE:Integer.MAX_VALUE;
		//find minimum
		for (BlockPos cc:in)
		{
			int curr = (coord==0?cc.func_177958_n():(coord==1?cc.func_177956_o():cc.func_177956_o()));
			if (max^(curr<currMinMax))
				currMinMax = curr;
		}
		//fill ret set
		for (BlockPos cc:in)
		{
			int curr = (coord==0?cc.func_177958_n():(coord==1?cc.func_177956_o():cc.func_177952_p()));
			if (curr==currMinMax)
				ret.add(cc);
		}
		return ret;
	}

	/**
	 * get tile entity without loading currently unloaded chunks
	 * @return return value of {@link IBlockAccess#getTileEntity(BlockPos)} or always null if chunk is not loaded
	 */
	public static TileEntity getExistingTileEntity(World world, BlockPos pos)
	{
		if(world.func_175667_e(pos))
			return world.func_175625_s(pos);
		return null;
	}
	public static NonNullList<ItemStack> readInventory(NBTTagList nbt, int size)
	{
		NonNullList<ItemStack> inv = NonNullList.func_191197_a(size, ItemStack.field_190927_a);
		int max = nbt.func_74745_c();
		for (int i = 0;i<max;i++)
		{
			NBTTagCompound itemTag = nbt.func_150305_b(i);
			int slot = itemTag.func_74771_c("Slot") & 255;
			if(slot>=0 && slot<size)
				inv.set(slot, new ItemStack(itemTag));
		}
		return inv;
	}
	public static NBTTagList writeInventory(ItemStack[] inv)
	{
		NBTTagList invList = new NBTTagList();
		for(int i=0; i<inv.length; i++)
			if(!inv[i].func_190926_b())
			{
				NBTTagCompound itemTag = new NBTTagCompound();
				itemTag.func_74774_a("Slot", (byte)i);
				inv[i].func_77955_b(itemTag);
				invList.func_74742_a(itemTag);
			}
		return invList;
	}
	public static NBTTagList writeInventory(Collection<ItemStack> inv)
	{
		NBTTagList invList = new NBTTagList();
		byte slot = 0;
		for(ItemStack s : inv)
		{
			if(!s.func_190926_b())
			{
				NBTTagCompound itemTag = new NBTTagCompound();
				itemTag.func_74774_a("Slot", slot);
				s.func_77955_b(itemTag);
				invList.func_74742_a(itemTag);
			}
			slot++;
		}
		return invList;
	}
	public static NonNullList<ItemStack> loadItemStacksFromNBT(NBTBase nbt)
	{
		NonNullList<ItemStack> itemStacks = NonNullList.func_191196_a();
		if(nbt instanceof NBTTagCompound)
		{
			ItemStack stack = new ItemStack((NBTTagCompound)nbt);
			itemStacks.add(stack);
			return itemStacks;
		}
		else if(nbt instanceof NBTTagList)
		{
			NBTTagList list = (NBTTagList)nbt;
			return readInventory(list, list.func_74745_c());
		}
		return itemStacks;
	}

	public static void modifyInvStackSize(NonNullList<ItemStack> inv, int slot, int amount)
	{
		if(slot>=0&&slot< inv.size() && !inv.get(slot).func_190926_b())
		{
			inv.get(slot).func_190917_f(amount);
			if(inv.get(slot).func_190916_E() <= 0)
				inv.set(slot, ItemStack.field_190927_a);
		}
	}

	public static void shuffleLootItems(List<ItemStack> stacks, int slotAmount, Random rand)
	{
		List<ItemStack> list = Lists.newArrayList();
		Iterator<ItemStack> iterator = stacks.iterator();
		while(iterator.hasNext())
		{
			ItemStack itemstack = iterator.next();
			if(itemstack.func_190916_E() <= 0)
				iterator.remove();
			else if(itemstack.func_190916_E() > 1)
			{
				list.add(itemstack);
				iterator.remove();
			}
		}
		slotAmount = slotAmount - stacks.size();
		while(slotAmount>0 && list.size()>0)
		{
			ItemStack itemstack2 = list.remove(MathHelper.func_76136_a(rand, 0, list.size() - 1));
			int i = MathHelper.func_76136_a(rand, 1, itemstack2.func_190916_E() / 2);
			itemstack2.func_190918_g(i);
			ItemStack itemstack1 = itemstack2.func_77946_l();
			itemstack1.func_190920_e(i);

			if(itemstack2.func_190916_E() > 1 && rand.nextBoolean())
				list.add(itemstack2);
			else
				stacks.add(itemstack2);

			if(itemstack1.func_190916_E() > 1 && rand.nextBoolean())
				list.add(itemstack1);
			else
				stacks.add(itemstack1);
		}
		stacks.addAll(list);
		Collections.shuffle(stacks, rand);
	}
	private static final Gson GSON_INSTANCE = (new GsonBuilder()).registerTypeAdapter(RandomValueRange.class, new RandomValueRange.Serializer()).registerTypeAdapter(LootPool.class, new LootPool.Serializer()).registerTypeAdapter(LootTable.class, new LootTable.Serializer()).registerTypeHierarchyAdapter(LootEntry.class, new LootEntry.Serializer()).registerTypeHierarchyAdapter(LootFunction.class, new LootFunctionManager.Serializer()).registerTypeHierarchyAdapter(LootCondition.class, new LootConditionManager.Serializer()).registerTypeHierarchyAdapter(LootContext.EntityTarget.class, new LootContext.EntityTarget.Serializer()).create();
	public static LootTable loadBuiltinLootTable(ResourceLocation resource, LootTableManager lootTableManager)
	{
		URL url = Utils.class.getResource("/assets/" + resource.func_110624_b() + "/loot_tables/" + resource.func_110623_a() + ".json");
		if(url==null)
			return LootTable.field_186464_a;
		else
		{
			String s;
			try
			{
				s = Resources.toString(url, Charsets.UTF_8);
			} catch(IOException ioexception)
			{
//				IELogger.warn(("Failed to load loot table " + resource.toString() + " from " + url.toString()));
				ioexception.printStackTrace();
				return LootTable.field_186464_a;
			}

			try
			{
				return net.minecraftforge.common.ForgeHooks.loadLootTable(GSON_INSTANCE, resource, s, false, lootTableManager);
			} catch(JsonParseException jsonparseexception)
			{
//				IELogger.error(("Failed to load loot table " + resource.toString() + " from " + url.toString()));
				jsonparseexception.printStackTrace();
				return LootTable.field_186464_a;
			}
		}
	}

	public static int calcRedstoneFromInventory(IIEInventory inv)
	{
		if(inv==null)
			return 0;
		else
		{
			int max = inv.getComparatedSize();
			int i=0;
			float f = 0.0F;
			for(int j = 0; j< max; ++j)
			{
				ItemStack itemstack = inv.getInventory().get(j);
				if(!itemstack.func_190926_b())
				{
					f += (float) itemstack.func_190916_E() / (float)Math.min(inv.getSlotLimit(j), itemstack.func_77976_d());
					++i;
				}
			}
			f = f/(float) max;
			return MathHelper.func_76141_d(f * 14.0F) + (i > 0 ? 1 : 0);
		}
	}


	public static Map<String, Object> saveStack(ItemStack stack)
	{
		HashMap<String, Object> ret = new HashMap<>();
		if(!stack.func_190926_b())
		{
			ret.put("size", stack.func_190916_E());
			ret.put("name", Item.field_150901_e.func_177774_c(stack.func_77973_b()));
			ret.put("nameUnlocalized", stack.func_77977_a());
			ret.put("label", stack.func_82833_r());
			ret.put("damage", stack.func_77952_i());
			ret.put("maxDamage", stack.func_77958_k());
			ret.put("maxSize", stack.func_77976_d());
			ret.put("hasTag", stack.func_77942_o());
		}
		return ret;
	}

	public static Map<String, Object> saveFluidTank(FluidTank tank)
	{
		HashMap<String, Object> ret = new HashMap<>();
		if(tank != null && tank.getFluid() != null)
		{
			ret.put("name", tank.getFluid().getFluid().getUnlocalizedName());
			ret.put("amount", tank.getFluidAmount());
			ret.put("capacity", tank.getCapacity());
			ret.put("hasTag", tank.getFluid().tag != null);
		}
		return ret;
	}

	public static Map<String, Object> saveFluidStack(FluidStack tank)
	{
		HashMap<String, Object> ret = new HashMap<>();
		if(tank != null && tank.getFluid() != null)
		{
			ret.put("name", tank.getFluid().getUnlocalizedName());
			ret.put("amount", tank.amount);
			ret.put("hasTag", tank.tag != null);
		}
		return ret;
	}

	public static void stateToNBT(NBTTagCompound out, IBlockState state)
	{
		out.func_74778_a("block", state.func_177230_c().getRegistryName().toString());
		for (IProperty<?> prop:state.func_177227_a())
			saveProp(state, prop, out);
	}

	public static IBlockState stateFromNBT(NBTTagCompound in)
	{
		Block b = Block.func_149684_b(in.func_74779_i("block"));
		if (b==null)
			return Blocks.field_150342_X.func_176223_P();
		IBlockState ret = b.func_176223_P();
		for (IProperty<?> prop:ret.func_177227_a())
		{
			String name = prop.func_177701_a();
			if (in.func_150297_b(name, Constants.NBT.TAG_STRING))
				ret = setProp(ret, prop, in.func_74779_i(name));
		}
		return ret;
	}

	private static <T extends Comparable<T>> void saveProp(IBlockState state, IProperty<T> prop, NBTTagCompound out)
	{
		out.func_74778_a(prop.func_177701_a(), prop.func_177702_a(state.func_177229_b(prop)));
	}

	private static <T extends Comparable<T>> IBlockState setProp(IBlockState state, IProperty<T> prop, String value)
	{
		Optional<T> valueParsed = prop.func_185929_b(value);
		if (valueParsed.isPresent())
			return state.func_177226_a(prop, valueParsed.get());
		return state;
	}
}
