/*
 * BluSunrize
 * Copyright (c) 2017
 *
 * This code is licensed under "Blu's License of Common Sense"
 * Details can be found in the license file in the root folder of this project
 */

package blusunrize.immersiveengineering.common.util;

import net.minecraft.item.ItemStack;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.nbt.NBTTagList;
import net.minecraft.nbt.NBTTagString;
import net.minecraftforge.fluids.FluidStack;

import java.util.regex.Pattern;

public class ItemNBTHelper
{
	public static NBTTagCompound getTag(ItemStack stack)
	{
		if(!stack.func_77942_o())
			stack.func_77982_d(new NBTTagCompound());
		return stack.func_77978_p();
	}
	public static boolean hasTag(ItemStack stack)
	{
		return stack.func_77942_o();
	}
	public static boolean hasKey(ItemStack stack, String key)
	{
		return hasTag(stack) && getTag(stack).func_74764_b(key);
	}

	public static void remove(ItemStack stack, String key)
	{
		if(hasKey(stack, key))
		{
			getTag(stack).func_82580_o(key);
			if(getTag(stack).func_82582_d())
				stack.func_77982_d(null);
		}
	}


	public static void setInt(ItemStack stack, String key, int val)
	{
		getTag(stack).func_74768_a(key, val);
	}
	public static void modifyInt(ItemStack stack, String key, int mod)
	{
		modifyInt(getTag(stack), key, mod);
	}
	public static void modifyInt(NBTTagCompound tagCompound, String key, int mod)
	{
		tagCompound.func_74768_a(key, tagCompound.func_74762_e(key)+mod);
	}
	public static int getInt(ItemStack stack, String key)
	{
		return hasTag(stack) ? getTag(stack).func_74762_e(key) : 0;
	}

	public static void setString(ItemStack stack, String key, String val)
	{
		getTag(stack).func_74778_a(key, val);
	}
	public static String getString(ItemStack stack, String key)
	{
		return hasTag(stack) ? getTag(stack).func_74779_i(key) : "";
	}

	public static void setLong(ItemStack stack, String key, long val)
	{
		getTag(stack).func_74772_a(key, val);
	}
	public static long getLong(ItemStack stack, String key)
	{
		return hasTag(stack) ? getTag(stack).func_74763_f(key) : 0;
	}

	public static void setIntArray(ItemStack stack, String key, int[] val)
	{
		getTag(stack).func_74783_a(key, val);
	}
	public static int[] getIntArray(ItemStack stack, String key)
	{
		return hasTag(stack) ? getTag(stack).func_74759_k(key) : new int[0];
	}

	public static void setFloat(ItemStack stack, String key, float val)
	{
		getTag(stack).func_74776_a(key, val);
	}
	public static void modifyFloat(NBTTagCompound tagCompound, String key, float mod)
	{
		tagCompound.func_74776_a(key, tagCompound.func_74760_g(key)+mod);
	}
	public static float getFloat(ItemStack stack, String key)
	{
		return hasTag(stack) ? getTag(stack).func_74760_g(key) : 0;
	}

	public static void setBoolean(ItemStack stack, String key, boolean val)
	{
		getTag(stack).func_74757_a(key, val);
	}
	public static boolean getBoolean(ItemStack stack, String key)
	{
		return hasTag(stack) && getTag(stack).func_74767_n(key);
	}

	public static void setTagCompound(ItemStack stack, String key, NBTTagCompound val)
	{
		getTag(stack).func_74782_a(key, val);
	}
	public static NBTTagCompound getTagCompound(ItemStack stack, String key)
	{
		return hasTag(stack) ? getTag(stack).func_74775_l(key) : new NBTTagCompound();
	}

	public static void setFluidStack(ItemStack stack, String key, FluidStack val)
	{
		if(val!=null && val.getFluid()!=null)
		{
			setTagCompound(stack, key, val.writeToNBT(new NBTTagCompound()));
		}
		else
			remove(stack, key);
	}
	public static FluidStack getFluidStack(ItemStack stack, String key)
	{
		if(hasTag(stack))
		{
			return FluidStack.loadFluidStackFromNBT(getTagCompound(stack, key));
		}
		return null;
	}

	public static void setItemStack(ItemStack stack, String key, ItemStack val)
	{
		getTag(stack).func_74782_a(key, val.func_77955_b(new NBTTagCompound()));
	}
	public static ItemStack getItemStack(ItemStack stack, String key)
	{
		if(hasTag(stack) && getTag(stack).func_74764_b(key))
			return new ItemStack(getTagCompound(stack, key));
		return ItemStack.field_190927_a;
	}

	public static void setLore(ItemStack stack, String... lore)
	{
		NBTTagCompound displayTag = getTagCompound(stack, "display");
		NBTTagList list = new NBTTagList();
		for(String s : lore)
			list.func_74742_a(new NBTTagString(s));
		displayTag.func_74782_a("Lore", list);
		setTagCompound(stack, "display", displayTag);
	}

	public static int insertFluxItem(ItemStack container, int energy, int maxEnergy, boolean simulate)
	{
		int stored = getFluxStoredInItem(container);
		int accepted = Math.min(energy, maxEnergy-stored);
		if(!simulate)
		{
			stored += accepted;
			ItemNBTHelper.setInt(container, "energy", stored);
		}
		return accepted;
	}
	public static int extractFluxFromItem(ItemStack container, int energy, boolean simulate)
	{
		int stored = getFluxStoredInItem(container);
		int extracted = Math.min(energy, stored);
		if(!simulate)
		{
			stored -= extracted;
			ItemNBTHelper.setInt(container, "energy", stored);
		}
		return extracted;
	}
	public static int getFluxStoredInItem(ItemStack container)
	{
		return getInt(container, "energy");
	}

	public static ItemStack stackWithData(ItemStack stack, Object... data)
	{
		assert(data.length%2==0);
		for(int i=0; i<data.length/2; i++)
		{
			Object key = data[i];
			Object value = data[i+1];
			if(key instanceof String)
			{
				if(value instanceof Boolean)
					setBoolean(stack, (String)key, (Boolean)value);
				else if(value instanceof Integer)
					setInt(stack, (String)key, (Integer)value);
				else if(value instanceof Float)
					setFloat(stack, (String)key, (Float)value);
				else if(value instanceof Long)
					setLong(stack, (String)key, (Long)value);
				else if(value instanceof String)
					setString(stack, (String)key, (String)value);
				else if(value instanceof NBTTagCompound)
					setTagCompound(stack, (String)key, (NBTTagCompound)value);
				else if(value instanceof int[])
					setIntArray(stack, (String)key, (int[])value);
				else if(value instanceof ItemStack)
					setItemStack(stack, (String)key, (ItemStack)value);
				else if(value instanceof FluidStack)
					setFluidStack(stack, (String)key, (FluidStack)value);
			}
		}
		return stack;
	}

	public static NBTTagCompound combineTags(NBTTagCompound target, NBTTagCompound add, Pattern pattern)
	{
		if(target==null || target.func_82582_d())
			return add.func_74737_b();
		for(String key : add.func_150296_c())
			if(pattern==null || pattern.matcher(key).matches())
				if(!target.func_74764_b(key))
					target.func_74782_a(key, add.func_74781_a(key));
				else
				{
					switch(add.func_150299_b(key))
					{
						case 1: //Byte
							target.func_74774_a(key, (byte)(target.func_74771_c(key)+add.func_74771_c(key)));
							break;
						case 2: //Short
							target.func_74777_a(key, (short)(target.func_74765_d(key)+add.func_74765_d(key)));
							break;
						case 3: //Int
							target.func_74768_a(key, (target.func_74762_e(key)+add.func_74762_e(key)));
							break;
						case 4: //Long
							target.func_74772_a(key, (target.func_74763_f(key)+add.func_74763_f(key)));
							break;
						case 5: //Float
							target.func_74776_a(key, (target.func_74760_g(key)+add.func_74760_g(key)));
							break;
						case 6: //Double
							target.func_74780_a(key, (target.func_74769_h(key)+add.func_74769_h(key)));
							break;
						case 7: //ByteArray
							byte[] bytesTarget = target.func_74770_j(key);
							byte[] bytesAdd = add.func_74770_j(key);
							byte[] bytes = new byte[bytesTarget.length+bytesAdd.length];
							System.arraycopy(bytesTarget,0, bytes,0, bytesTarget.length);
							System.arraycopy(bytesAdd,0, bytes,bytesTarget.length, bytesAdd.length);
							target.func_74773_a(key, bytes);
							break;
						case 8: //String
							target.func_74778_a(key, (target.func_74779_i(key)+add.func_74779_i(key)));
							break;
						case 9: //List
							NBTTagList listTarget = (NBTTagList)target.func_74781_a(key);
							NBTTagList listAdd = (NBTTagList)add.func_74781_a(key);
							for(int i=0; i<listAdd.func_74745_c(); i++)
								listTarget.func_74742_a(listAdd.func_179238_g(i));
							target.func_74782_a(key, listTarget);
							break;
						case 10: //Compound
							combineTags(target.func_74775_l(key), add.func_74775_l(key), null);
							break;
						case 11: //IntArray
							int[] intsTarget = target.func_74759_k(key);
							int[] intsAdd = add.func_74759_k(key);
							int[] ints = new int[intsTarget.length+intsAdd.length];
							System.arraycopy(intsTarget,0, ints,0, intsTarget.length);
							System.arraycopy(intsAdd,0, ints,intsTarget.length, intsAdd.length);
							target.func_74783_a(key, ints);
							break;
					}
				}
		return target;
	}
}
