/*
 * BluSunrize
 * Copyright (c) 2017
 *
 * This code is licensed under "Blu's License of Common Sense"
 * Details can be found in the license file in the root folder of this project
 */

package blusunrize.immersiveengineering.common.items;

import blusunrize.immersiveengineering.api.Lib;
import blusunrize.immersiveengineering.api.tool.BulletHandler;
import blusunrize.immersiveengineering.api.tool.BulletHandler.IBullet;
import blusunrize.immersiveengineering.client.ClientProxy;
import blusunrize.immersiveengineering.common.Config.IEConfig;
import blusunrize.immersiveengineering.common.entities.EntityRevolvershot;
import blusunrize.immersiveengineering.common.entities.EntityRevolvershotFlare;
import blusunrize.immersiveengineering.common.entities.EntityRevolvershotHoming;
import blusunrize.immersiveengineering.common.entities.EntityWolfpackShot;
import blusunrize.immersiveengineering.common.items.IEItemInterfaces.IColouredItem;
import blusunrize.immersiveengineering.common.items.IEItemInterfaces.ITextureOverride;
import blusunrize.immersiveengineering.common.util.IEDamageSources;
import blusunrize.immersiveengineering.common.util.IESounds;
import blusunrize.immersiveengineering.common.util.ItemNBTHelper;
import blusunrize.immersiveengineering.common.util.Utils;
import blusunrize.immersiveengineering.common.util.chickenbones.Matrix4;
import net.minecraft.client.gui.FontRenderer;
import net.minecraft.client.util.ITooltipFlag;
import net.minecraft.creativetab.CreativeTabs;
import net.minecraft.entity.Entity;
import net.minecraft.entity.EntityAreaEffectCloud;
import net.minecraft.entity.EntityLivingBase;
import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.item.ItemLingeringPotion;
import net.minecraft.item.ItemPotion;
import net.minecraft.item.ItemSplashPotion;
import net.minecraft.item.ItemStack;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.potion.PotionEffect;
import net.minecraft.potion.PotionType;
import net.minecraft.potion.PotionUtils;
import net.minecraft.util.*;
import net.minecraft.util.math.BlockPos;
import net.minecraft.util.math.RayTraceResult;
import net.minecraft.util.math.Vec3d;
import net.minecraft.util.text.translation.I18n;
import net.minecraft.world.World;
import net.minecraftforge.fml.relauncher.Side;
import net.minecraftforge.fml.relauncher.SideOnly;

import javax.annotation.Nullable;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.function.Function;

public class ItemBullet extends ItemIEBase implements ITextureOverride//IBullet
{
	public ItemBullet()
	{
//		super("bullet", 64, "emptyCasing","emptyShell","casull","armorPiercing","buckshot","HE","dragonsbreath","homing","wolfpack","silver","potion","flare");
		super("bullet", 64, "empty_casing", "empty_shell", "bullet");

		BulletHandler.emptyCasing = new ItemStack(this, 1, 0);
		BulletHandler.emptyShell = new ItemStack(this, 1, 1);
		BulletHandler.basicCartridge = new ItemStack(this, 1, 2);
	}

	public static void initBullets()
	{
		BulletHandler.registerBullet("casull", new BulletHandler.DamagingBullet(
				entities -> IEDamageSources.causeCasullDamage((EntityRevolvershot)entities[0], entities[1]),
				IEConfig.Tools.bulletDamage_Casull,
				BulletHandler.emptyCasing,
				new ResourceLocation("immersiveengineering:items/bullet_casull")));

		BulletHandler.registerBullet("armor_piercing", new BulletHandler.DamagingBullet(
				entities -> IEDamageSources.causePiercingDamage((EntityRevolvershot)entities[0], entities[1]),
				IEConfig.Tools.bulletDamage_AP,
				BulletHandler.emptyCasing,
				new ResourceLocation("immersiveengineering:items/bullet_armor_piercing")));

		BulletHandler.registerBullet("buckshot", new BulletHandler.DamagingBullet(
				entities -> IEDamageSources.causeBuckshotDamage((EntityRevolvershot)entities[0], entities[1]),
				IEConfig.Tools.bulletDamage_Buck,
				true,
				false,
				BulletHandler.emptyShell,
				new ResourceLocation("immersiveengineering:items/bullet_buckshot"))
		{
			@Override
			public int getProjectileCount(EntityPlayer shooter)
			{
				return 10;
			}
		});

		BulletHandler.registerBullet("he", new BulletHandler.DamagingBullet(null, 0, BulletHandler.emptyCasing, new ResourceLocation("immersiveengineering:items/bullet_he"))
		{
			@Override
			public void onHitTarget(World world, RayTraceResult target, EntityLivingBase shooter, Entity projectile, boolean headshot)
			{
				world.func_72876_a(shooter, projectile.field_70165_t, projectile.field_70163_u, projectile.field_70161_v, 2, false);
			}

			@Override
			public Entity getProjectile(@Nullable EntityPlayer shooter, ItemStack cartridge, Entity projectile, boolean charged)
			{
				if(projectile instanceof EntityRevolvershot)
				{
					((EntityRevolvershot)projectile).setGravity(0.05f);
					((EntityRevolvershot)projectile).setMovementDecay(0.9f);
				}
				return projectile;
			}

			@Override
			public SoundEvent getSound()
			{
				return IESounds.revolverFireThump;
			}
		});

		BulletHandler.registerBullet("silver", new BulletHandler.DamagingBullet(
				entities -> IEDamageSources.causeSilverDamage((EntityRevolvershot)entities[0], entities[1]),
				IEConfig.Tools.bulletDamage_Silver,
				BulletHandler.emptyCasing,
				new ResourceLocation("immersiveengineering:items/bullet_silver")));

		BulletHandler.registerBullet("dragonsbreath", new BulletHandler.DamagingBullet(
				entities -> IEDamageSources.causeDragonsbreathDamage((EntityRevolvershot)entities[0], entities[1]),
				IEConfig.Tools.bulletDamage_Dragon,
				true,
				true,
				BulletHandler.emptyShell,
				new ResourceLocation("immersiveengineering:items/bullet_dragonsbreath"))
		{
			@Override
			public int getProjectileCount(EntityPlayer shooter)
			{
				return 30;
			}

			@Override
			public Entity getProjectile(EntityPlayer shooter, ItemStack cartridge, Entity projectile, boolean electro)
			{
				((EntityRevolvershot)projectile).setTickLimit(10);
				projectile.func_70015_d(3);
				return projectile;
			}
		});

		BulletHandler.registerBullet("potion", new PotionBullet());

		BulletHandler.registerBullet("flare", new FlareBullet());
	}

	@Override
	@SideOnly(Side.CLIENT)
	public void func_150895_a(CreativeTabs tab, NonNullList<ItemStack> list)
	{
		if(this.func_194125_a(tab))
		{
			list.add(new ItemStack(this, 1, 0));
			list.add(new ItemStack(this, 1, 1));
			for(Map.Entry<String, IBullet> entry : BulletHandler.registry.entrySet())
				if(entry.getValue().isProperCartridge())
				{
					ItemStack s = new ItemStack(this, 1, 2);
					ItemNBTHelper.setString(s, "bullet", entry.getKey());
					list.add(s);
				}
		}
	}

	@Override
	@SideOnly(Side.CLIENT)
	public FontRenderer getFontRenderer(ItemStack stack)
	{
		return ClientProxy.itemFont;
	}

	@SideOnly(Side.CLIENT)
	@Override
	public void func_77624_a(ItemStack stack, @Nullable World world, List<String> list, ITooltipFlag flag)
	{
		if(stack.func_77952_i() == 2)
		{
			String key = ItemNBTHelper.getString(stack, "bullet");
			IBullet bullet = BulletHandler.getBullet(key);
			if(bullet != null)
				bullet.addTooltip(stack, world, list, flag);
		}
	}
	@Override
	public String func_77653_i(ItemStack stack)
	{
		if(stack.func_77952_i() == 2)
		{
			String s = "item.immersiveengineering.bullet.";
			String key = ItemNBTHelper.getString(stack, "bullet");
			// handle legacy bullets
			key = BulletHandler.handleLeagcyNames(key);
			s += key;
			IBullet bullet = BulletHandler.getBullet(key);
			if(bullet != null)
				s = bullet.getUnlocalizedName(stack, s);
			return I18n.func_74838_a(s + ".name").trim();
		}
		return super.func_77653_i(stack);
	}

	@Override
	public boolean hasCustomItemColours()
	{
		return true;
	}
	@Override
	public int getColourForIEItem(ItemStack stack, int pass)
	{
		if(stack.func_77960_j() == 2 && ItemNBTHelper.hasKey(stack, "bullet"))
		{
			IBullet bullet = BulletHandler.getBullet(ItemNBTHelper.getString(stack, "bullet"));
			if(bullet != null)
				return bullet.getColour(stack, pass);
		}
		return super.getColourForIEItem(stack, pass);
	}
	@Override
	@SideOnly(Side.CLIENT)
	public String getModelCacheKey(ItemStack stack)
	{
		if(stack.func_77960_j() == 2 && ItemNBTHelper.hasKey(stack, "bullet"))
			return ItemNBTHelper.getString(stack, "bullet");
		return null;
	}
	@Override
	@SideOnly(Side.CLIENT)
	public List<ResourceLocation> getTextures(ItemStack stack, String key)
	{
		IBullet bullet = BulletHandler.getBullet(key);
		if(bullet != null)
			return Arrays.asList(bullet.getTextures());
		return Arrays.asList(new ResourceLocation("immersiveengieering:items/bullet_casull"));
	}

	//	@Override
//	public ItemStack getCasing(ItemStack stack)
//	{
//		return new ItemStack(this, 1, stack.getItemDamage()==1||stack.getItemDamage()==4||stack.getItemDamage()==6||stack.getItemDamage()==11?1:0);
//	}
//	@Override
//	public boolean canSpawnBullet(ItemStack bulletStack)
//	{
//		return bulletStack!=null && bulletStack.getItemDamage()>1 && (bulletStack.getItemDamage()!=10||ItemNBTHelper.getItemStack(bulletStack, "potion")!=null);
//	}
//	@Override
//	public void spawnBullet(EntityPlayer player, ItemStack bulletStack, boolean electro)
//	{
//		Vec3d vec = player.getLookVec();
//		int type = bulletStack.getItemDamage()-2;
//		switch(type)
//		{
//			case 0://casull
//				doSpawnBullet(player, vec, vec, type, bulletStack, electro);
//				break;
//			case 1://armorPiercing
//				doSpawnBullet(player, vec, vec, type, bulletStack, electro);
//				break;
//			case 2://buckshot
//				for(int i=0; i<10; i++)
//				{
//					Vec3d vecDir = vec.addVector(player.getRNG().nextGaussian()*.1,player.getRNG().nextGaussian()*.1,player.getRNG().nextGaussian()*.1);
//					doSpawnBullet(player, vec, vecDir, type, bulletStack, electro);
//				}
//				break;
//			case 3://HE
//				doSpawnBullet(player, vec, vec, type, bulletStack, electro);
//				break;
//			case 4://dragonsbreath
//				for(int i=0; i<30; i++)
//				{
//					Vec3d vecDir = vec.addVector(player.getRNG().nextGaussian()*.1,player.getRNG().nextGaussian()*.1,player.getRNG().nextGaussian()*.1);
//					EntityRevolvershot shot = doSpawnBullet(player, vec, vecDir, type, bulletStack, electro);
//					shot.setTickLimit(10);
//					shot.setFire(3);
//				}
//				break;
//			case 5://homing
//				EntityRevolvershotHoming bullet = new EntityRevolvershotHoming(player.world, player, vec.xCoord*1.5,vec.yCoord*1.5,vec.zCoord*1.5, type, bulletStack);
//				bullet.motionX = vec.xCoord;
//				bullet.motionY = vec.yCoord;
//				bullet.motionZ = vec.zCoord;
//				bullet.bulletElectro = electro;
//				player.world.spawnEntity(bullet);
//				break;
//			case 6://wolfpack
//				doSpawnBullet(player, vec, vec, type, bulletStack, electro);
//				break;
//			case 7://Silver
//				doSpawnBullet(player, vec, vec, type, bulletStack, electro);
//				break;
//			case 8://Potion
//				EntityRevolvershot shot = doSpawnBullet(player, vec, vec, type, bulletStack, electro);
//				shot.bulletPotion = ItemNBTHelper.getItemStack(bulletStack, "potion");
//				break;
//			case 9://Flare
//				EntityRevolvershotFlare flare = new EntityRevolvershotFlare(player.world, player, vec.xCoord*1.5,vec.yCoord*1.5,vec.zCoord*1.5, type, bulletStack);
//				flare.motionX = vec.xCoord;
//				flare.motionY = vec.yCoord;
//				flare.motionZ = vec.zCoord;
//				flare.bulletElectro = electro;
//				flare.colour = this.getColourForIEItem(bulletStack, 1);
//				flare.setColourSynced();
//				player.world.spawnEntity(flare);
//				break;
//		}
//	}


	public static class PotionBullet extends BulletHandler.DamagingBullet
	{
		public PotionBullet()
		{
			super(entities -> IEDamageSources.causePotionDamage((EntityRevolvershot) entities[0], entities[1]),
					IEConfig.Tools.bulletDamage_Potion,
					BulletHandler.emptyCasing,
					new ResourceLocation("immersiveengineering:items/bullet_potion"), new ResourceLocation("immersiveengineering:items/bullet_potion_layer"));
		}

		@Override
		public String getUnlocalizedName(ItemStack cartridge, String baseName)
		{
			ItemStack pot = ItemNBTHelper.getItemStack(cartridge, "potion");
			if(!pot.func_190926_b())
				if(pot.func_77973_b() instanceof ItemLingeringPotion)
					baseName += ".linger";
				else if(pot.func_77973_b() instanceof ItemSplashPotion)
					baseName += ".splash";
			return baseName;
		}

		@Override
		public Entity getProjectile(EntityPlayer shooter, ItemStack cartridge, Entity projectile, boolean electro)
		{
			((EntityRevolvershot) projectile).bulletPotion = ItemNBTHelper.getItemStack(cartridge, "potion");
			return projectile;
		}

		@Override
		public void onHitTarget(World world, RayTraceResult target, EntityLivingBase shooter, Entity projectile, boolean headshot)
		{
			super.onHitTarget(world, target, shooter, projectile, headshot);
			EntityRevolvershot bullet = (EntityRevolvershot)projectile;
			if(!bullet.bulletPotion.func_190926_b() && bullet.bulletPotion.func_77942_o())
			{
				PotionType potionType = PotionUtils.func_185191_c(bullet.bulletPotion);
				List<PotionEffect> effects = PotionUtils.func_185189_a(bullet.bulletPotion);
				if(effects != null)
					if(bullet.bulletPotion.func_77973_b() instanceof ItemLingeringPotion)
					{
						EntityAreaEffectCloud entityareaeffectcloud = new EntityAreaEffectCloud(bullet.field_70170_p, bullet.field_70165_t, bullet.field_70163_u, bullet.field_70161_v);
						entityareaeffectcloud.func_184481_a(shooter);
						entityareaeffectcloud.func_184483_a(3.0F);
						entityareaeffectcloud.func_184495_b(-0.5F);
						entityareaeffectcloud.func_184485_d(10);
						entityareaeffectcloud.func_184487_c(-entityareaeffectcloud.func_184490_j() / (float)entityareaeffectcloud.func_184489_o());
						entityareaeffectcloud.func_184484_a(potionType);
						for(PotionEffect potioneffect : effects)
							entityareaeffectcloud.func_184496_a(new PotionEffect(potioneffect.func_188419_a(), potioneffect.func_76459_b(), potioneffect.func_76458_c()));
						bullet.field_70170_p.func_72838_d(entityareaeffectcloud);
					}
					else if(bullet.bulletPotion.func_77973_b() instanceof ItemSplashPotion)
					{
						List<EntityLivingBase> livingEntities = bullet.field_70170_p.func_72872_a(EntityLivingBase.class, bullet.func_174813_aQ().func_72314_b(4.0D, 2.0D, 4.0D));
						if(livingEntities != null && !livingEntities.isEmpty())
							for(EntityLivingBase living : livingEntities)
								if(living.func_184603_cC())
								{
									double dist = bullet.func_70068_e(living);
									if(dist < 16D)
									{
										double dist2 = 1 - Math.sqrt(dist) / 4D;
										if(living == target.field_72308_g)
											dist2 = 1D;
										for(PotionEffect p : effects)
											if(p.func_188419_a().func_76403_b())
												p.func_188419_a().func_180793_a(bullet, shooter, living, p.func_76458_c(), dist2);
											else
											{
												int j = (int)(dist2 * p.func_76459_b() + .5D);
												if(j > 20)
													living.func_70690_d(new PotionEffect(p.func_188419_a(), j, p.func_76458_c()));
											}
									}
								}

					}
					else if(target.field_72308_g instanceof EntityLivingBase)
						for(PotionEffect p : effects)
						{
							if(p.func_76459_b() < 1)
								p = new PotionEffect(p.func_188419_a(), 1);
							((EntityLivingBase)target.field_72308_g).func_70690_d(p);
						}
				world.func_175718_b(2002, new BlockPos(bullet), PotionUtils.func_185183_a(potionType));
			}
		}


		@Override
		public void addTooltip(ItemStack stack, World world, List<String> list, ITooltipFlag flag)
		{
			ItemStack pot = ItemNBTHelper.getItemStack(stack, "potion");
			if(!pot.func_190926_b() && pot.func_77973_b() instanceof ItemPotion)
				PotionUtils.func_185182_a(pot, list, 1f);
		}

		@Override
		public int getColour(ItemStack stack, int layer)
		{
			if(layer == 1)
			{
				ItemStack pot = ItemNBTHelper.getItemStack(stack, "potion");
				return pot.func_190926_b() ? 0xff385dc6 : PotionUtils.func_185181_a(PotionUtils.func_185189_a(pot));
			}
			return 0xffffffff;
		}
	}

	public static class FlareBullet implements BulletHandler.IBullet
	{
		static ResourceLocation[] textures = {new ResourceLocation("immersiveengineering:items/bullet_flare"), new ResourceLocation("immersiveengineering:items/bullet_flare_layer")};

		public FlareBullet()
		{
		}

		@Override
		public Entity getProjectile(EntityPlayer shooter, ItemStack cartridge, Entity projectile, boolean electro)
		{
			EntityRevolvershotFlare flare =  shooter!=null?new EntityRevolvershotFlare(projectile.field_70170_p, shooter, projectile.field_70159_w*1.5,projectile.field_70181_x*1.5,projectile.field_70179_y*1.5, this, cartridge): new EntityRevolvershotFlare(projectile.field_70170_p, projectile.field_70165_t,projectile.field_70163_u,projectile.field_70161_v, 0,0,0, this);
			flare.field_70159_w = projectile.field_70159_w;
			flare.field_70181_x = projectile.field_70181_x;
			flare.field_70179_y = projectile.field_70179_y;
			flare.bulletElectro = electro;
			flare.colour = this.getColour(cartridge, 1);
			flare.setColourSynced();
			return flare;
		}

		@Override
		public void onHitTarget(World world, RayTraceResult target, EntityLivingBase shooter, Entity projectile, boolean headshot)
		{
		}

		@Override
		public ItemStack getCasing(ItemStack stack)
		{
			return BulletHandler.emptyShell;
		}

		@Override
		public ResourceLocation[] getTextures()
		{
			return textures;
		}

		@Override
		public void addTooltip(ItemStack stack, World world, List<String> list, ITooltipFlag flag)
		{
			if(stack.func_77973_b() instanceof IColouredItem)
			{
				String hexCol = Integer.toHexString(((IColouredItem)stack.func_77973_b()).getColourForIEItem(stack, 1));
				list.add(I18n.func_74837_a(Lib.DESC_INFO + "bullet.flareColour", "<hexcol=" + hexCol + ":#" + hexCol + ">"));
			}
		}

		@Override
		public int getColour(ItemStack stack, int layer)
		{
			if(layer != 1)
				return 0xffffffff;
			return ItemNBTHelper.hasKey(stack, "flareColour") ? ItemNBTHelper.getInt(stack, "flareColour") : 0xcc2e06;
		}

		@Override
		public boolean isValidForTurret()
		{
			return true;
		}
	}

	public static class HomingBullet extends BulletHandler.DamagingBullet
	{
		public HomingBullet(float damage, ResourceLocation... textures)
		{
			super(new Function<Entity[], DamageSource>()
				  {
					  @Override
					  public DamageSource apply(Entity[] entities)
					  {
						  return IEDamageSources.causeHomingDamage((EntityRevolvershot)entities[0], entities[1]);
					  }
				  },
					damage,
					BulletHandler.emptyCasing,
					textures);
		}

		@Override
		public Entity getProjectile(EntityPlayer shooter, ItemStack cartridge, Entity projectile, boolean electro)
		{
			EntityRevolvershotHoming shot = shooter!=null?new EntityRevolvershotHoming(projectile.field_70170_p, shooter, projectile.field_70159_w*1.5,projectile.field_70181_x*1.5,projectile.field_70179_y*1.5, this, cartridge): new EntityRevolvershotHoming(projectile.field_70170_p, projectile.field_70165_t,projectile.field_70163_u,projectile.field_70161_v, 0,0,0, this);
			shot.field_70159_w = projectile.field_70159_w;
			shot.field_70181_x = projectile.field_70181_x;
			shot.field_70179_y = projectile.field_70179_y;
			shot.bulletElectro = electro;
			return shot;
		}

		@Override
		public int getColour(ItemStack stack, int layer)
		{
			return 0xffffffff;
		}
	}

	public static class WolfpackBullet extends BulletHandler.DamagingBullet
	{
		public WolfpackBullet()
		{
			super(entities -> IEDamageSources.causeWolfpackDamage((EntityRevolvershot)entities[0], entities[1]),
					IEConfig.Tools.bulletDamage_Wolfpack,
					BulletHandler.emptyShell,
					new ResourceLocation("immersiveengineering:items/bullet_wolfpack"));
		}

		@Override
		public void onHitTarget(World world, RayTraceResult target, EntityLivingBase shooter, Entity projectile, boolean headshot)
		{
			super.onHitTarget(world, target, shooter, projectile, headshot);
			Vec3d v = new Vec3d(-projectile.field_70159_w, -projectile.field_70181_x, -projectile.field_70179_y);
			int split = 6;
			for(int i = 0; i < split; i++)
			{
				float angle = i * (360f / split);
				Matrix4 matrix = new Matrix4();
				matrix.rotate(angle, v.field_72450_a, v.field_72448_b, v.field_72449_c);
				Vec3d vecDir = new Vec3d(0, 1, 0);
				vecDir = matrix.apply(vecDir);

				EntityWolfpackShot bullet = shooter!=null?new EntityWolfpackShot(world, shooter, vecDir.field_72450_a * 1.5, vecDir.field_72448_b * 1.5, vecDir.field_72449_c * 1.5, this, null) : new EntityWolfpackShot(world, 0,0,0, 0,0,0, this);
				if(target.field_72308_g instanceof EntityLivingBase)
					bullet.targetOverride = (EntityLivingBase)target.field_72308_g;
				bullet.func_70107_b(target.field_72307_f.field_72450_a + vecDir.field_72450_a, target.field_72307_f.field_72448_b + vecDir.field_72448_b, target.field_72307_f.field_72449_c + vecDir.field_72449_c);
				bullet.field_70159_w = vecDir.field_72450_a * .375;
				bullet.field_70181_x = vecDir.field_72448_b * .375;
				bullet.field_70179_y = vecDir.field_72449_c * .375;
				world.func_72838_d(bullet);
			}
		}

		@Override
		public int getColour(ItemStack stack, int layer)
		{
			return 0xffffffff;
		}
	}

	public static class WolfpackPartBullet extends BulletHandler.DamagingBullet
	{
		public WolfpackPartBullet()
		{
			super(entities -> IEDamageSources.causeWolfpackDamage((EntityRevolvershot)entities[0], entities[1]),
					IEConfig.Tools.bulletDamage_WolfpackPart,
					BulletHandler.emptyCasing,
					new ResourceLocation("immersiveengineering:items/bullet_wolfpack"));
		}

		@Override
		public boolean isProperCartridge()
		{
			return false;
		}

		@Override
		public int getColour(ItemStack stack, int layer)
		{
			return 0xffffffff;
		}
	}
}
