/*
 * BluSunrize
 * Copyright (c) 2017
 *
 * This code is licensed under "Blu's License of Common Sense"
 * Details can be found in the license file in the root folder of this project
 */

package blusunrize.immersiveengineering.api.tool;

import blusunrize.immersiveengineering.api.ApiUtils;
import blusunrize.immersiveengineering.common.util.Utils;
import com.google.common.collect.Lists;
import net.minecraft.block.Block;
import net.minecraft.client.renderer.block.model.BakedQuad;
import net.minecraft.entity.Entity;
import net.minecraft.entity.item.EntityItem;
import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.item.ItemStack;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.tileentity.TileEntity;
import net.minecraft.util.EnumFacing;
import net.minecraft.util.EnumFacing.Axis;
import net.minecraft.util.EnumHand;
import net.minecraft.util.ResourceLocation;
import net.minecraft.util.math.AxisAlignedBB;
import net.minecraft.util.math.BlockPos;
import net.minecraft.util.math.Vec3d;
import net.minecraft.world.World;
import net.minecraftforge.fml.relauncher.Side;
import net.minecraftforge.fml.relauncher.SideOnly;

import javax.annotation.Nullable;
import javax.vecmath.Matrix4f;
import java.util.*;
import java.util.function.BiConsumer;
import java.util.function.Function;


/**
 * @author BluSunrize - 17.08.2016
 *         A handler for custom conveyor types
 */
public class ConveyorHandler
{
	public static HashMap<ResourceLocation, Class<? extends IConveyorBelt>> classRegistry = new LinkedHashMap<ResourceLocation, Class<? extends IConveyorBelt>>();
	public static HashMap<ResourceLocation, Function<TileEntity, ? extends IConveyorBelt>> functionRegistry = new LinkedHashMap<ResourceLocation, Function<TileEntity, ? extends IConveyorBelt>>();
	public static HashMap<Class<? extends IConveyorBelt>, ResourceLocation> reverseClassRegistry = new LinkedHashMap<Class<? extends IConveyorBelt>, ResourceLocation>();
	public static Set<BiConsumer<Entity, IConveyorTile>> magnetSupressionFunctions = new HashSet<BiConsumer<Entity, IConveyorTile>>();
	public static Set<BiConsumer<Entity, IConveyorTile>> magnetSupressionReverse = new HashSet<BiConsumer<Entity, IConveyorTile>>();

	public static Block conveyorBlock;
	public static ResourceLocation textureConveyorColour = new ResourceLocation("immersiveengineering:blocks/conveyor_colour");

	/**
	 * @param key           A unique ResourceLocation to identify the conveyor by
	 * @param conveyorClass the conveyor class
	 * @param function      a function used to create a new instance. Note that the TileEntity may be null for the inventory model. Handle accordingly.
	 */
	public static <T extends IConveyorBelt> boolean registerConveyorHandler(ResourceLocation key, Class<T> conveyorClass, Function<TileEntity, T> function)
	{
		if(classRegistry.containsKey(key))
			return false;
		classRegistry.put(key, conveyorClass);
		reverseClassRegistry.put(conveyorClass, key);
		functionRegistry.put(key, function);
		return true;
	}

	/**
	 * @return a new instance of the given conveyor type
	 */
	public static IConveyorBelt getConveyor(ResourceLocation key, @Nullable TileEntity tile)
	{
		Function<TileEntity, ? extends IConveyorBelt> func = functionRegistry.get(key);
		if(func != null)
			return func.apply(tile);
		return null;
	}

	/**
	 * @return an ItemStack with the given key written to NBT
	 */
	public static ItemStack getConveyorStack(String key)
	{
		ItemStack stack = new ItemStack(conveyorBlock);
		stack.func_77982_d(new NBTTagCompound());
		stack.func_77978_p().func_74778_a("conveyorType", key);
		return stack;
	}

	/**
	 * @return whether the given subtype key can be found at the location. Useful for multiblocks
	 */
	public static boolean isConveyor(World world, BlockPos pos, String key, @Nullable EnumFacing facing)
	{
		TileEntity tile = world.func_175625_s(pos);
		if(!(tile instanceof IConveyorTile))
			return false;
		if(facing != null && !facing.equals(((IConveyorTile) tile).getFacing()))
			return false;
		IConveyorBelt conveyor = ((IConveyorTile) tile).getConveyorSubtype();
		if(conveyor == null)
			return false;
		ResourceLocation rl = reverseClassRegistry.get(conveyor.getClass());
		return !(rl == null || !key.equalsIgnoreCase(rl.toString()));
	}

	/**
	 * registers a consumer/function to suppress magnets while they are on the conveyors
	 * the reversal function is optional, to revert possible NBT changes
	 * the tileentity parsed is an instanceof
	 */
	public static void registerMagnetSupression(BiConsumer<Entity, IConveyorTile> function, @Nullable BiConsumer<Entity, IConveyorTile> revert)
	{
		magnetSupressionFunctions.add(function);
		if(revert != null)
			magnetSupressionReverse.add(revert);
	}

	/**
	 * applies all registered magnets supressors to the entity
	 */
	public static void applyMagnetSupression(Entity entity, IConveyorTile tile)
	{
		if(entity != null)
			for(BiConsumer<Entity, IConveyorTile> func : magnetSupressionFunctions)
				func.accept(entity, tile);
	}

	/**
	 * applies all registered magnet supression removals
	 */
	public static void revertMagnetSupression(Entity entity, IConveyorTile tile)
	{
		if(entity != null)
			for(BiConsumer<Entity, IConveyorTile> func : magnetSupressionReverse)
				func.accept(entity, tile);
	}

	/**
	 * An interface for the external handling of conveyorbelts
	 */
	public interface IConveyorBelt
	{
		/**
		 * @return the string by which unique models would be cached. Override for additional appended information*
		 * The model class will also append to this key for rendered walls and facing
		 */
		default String getModelCacheKey(TileEntity tile, EnumFacing facing)
		{
			String key = reverseClassRegistry.get(this.getClass()).toString();
			key += "f" + facing.ordinal();
			key += "d" + getConveyorDirection().ordinal();
			key += "a" + (isActive(tile) ? 1 : 0);
			key += "w0" + (renderWall(tile, facing, 0) ? 1 : 0);
			key += "w1" + (renderWall(tile, facing, 1) ? 1 : 0);
			key += "c" + getDyeColour();
			return key;
		}

		/**
		 * @return the transport direction; HORIZONTAL for flat conveyors, UP and DOWN for diagonals
		 */
		default ConveyorDirection getConveyorDirection()
		{
			return ConveyorDirection.HORIZONTAL;
		}

		/**
		 * Switch to the next possible ConveyorDirection
		 * @return true if renderupdate should happen
		 */
		boolean changeConveyorDirection();

		/**
		 * Set the ConveyorDirection to given
		 *
		 * @return false if the direction is not possible for this conveyor
		 */
		boolean setConveyorDirection(ConveyorDirection dir);

		/**
		 * Called after the conveyor has been rotated with a hammer
		 */
		default void afterRotation(EnumFacing oldDir, EnumFacing newDir)
		{
		}

		/**
		 * @return false if the conveyor is deactivated (for instance by a redstone signal)
		 */
		boolean isActive(TileEntity tile);

		/**
		 * @return true if the conveyor can be dyed
		 */
		boolean canBeDyed();

		/**
		 * sets the colour of the conveyor when rightclicked with a dye
		 * parsed value is a hex RGB
		 *
		 * @return true if renderupdate should happen
		 */
		boolean setDyeColour(int colour);

		/**
		 * @return the dyed colour as a hex RGB
		 */
		int getDyeColour();

		/**
		 * when the player rightclicks the block, after direction changes or dye have been handled
		 *
		 * @return true if anything happened, cancelling item use
		 */
		default boolean playerInteraction(TileEntity tile, EntityPlayer player, EnumHand hand, ItemStack heldItem, float hitX, float hitY, float hitZ, EnumFacing side)
		{
			return false;
		}

		/**
		 * @param wall 0 is left, 1 is right
		 * @return whether the wall should be drawn on the model. Also used for they cache key
		 */
		default boolean renderWall(TileEntity tile, EnumFacing facing, int wall)
		{
			if(getConveyorDirection() != ConveyorDirection.HORIZONTAL)
				return true;
			EnumFacing side = wall == 0 ? facing.func_176735_f() : facing.func_176746_e();
			BlockPos pos = tile.func_174877_v().func_177972_a(side);
			TileEntity te = Utils.getExistingTileEntity(tile.func_145831_w(), pos);
			if(te instanceof IConveyorAttachable)
			{
				boolean b = false;
				for(EnumFacing f : ((IConveyorAttachable)te).sigOutputDirections())
					if(f == side.func_176734_d())
						b = true;
					else if(f == EnumFacing.UP)
						b = false;
				return !b;
			}
			else
			{
				te = Utils.getExistingTileEntity(tile.func_145831_w(), pos.func_177982_a(0, -1, 0));
				if(te instanceof IConveyorAttachable)
				{
					int b = 0;
					for(EnumFacing f : ((IConveyorAttachable)te).sigOutputDirections())
						if(f == side.func_176734_d())
							b++;
						else if(f == EnumFacing.UP)
							b++;
					return b < 2;
				}
			}
			return true;
		}

		/**
		 * a rough indication of where this conveyor will transport things. Relevant for vertical conveyors, to see if they need to render the groundpiece below them.
		 */
		default EnumFacing[] sigTransportDirections(TileEntity conveyorTile, EnumFacing facing)
		{
			if(getConveyorDirection() == ConveyorDirection.UP)
				return new EnumFacing[]{facing, EnumFacing.UP};
			else if(getConveyorDirection() == ConveyorDirection.DOWN)
				return new EnumFacing[]{facing, EnumFacing.DOWN};
			return new EnumFacing[]{facing};
		}

		/**
		 * @return a vector representing the movement applied to the entity
		 */
		default Vec3d getDirection(TileEntity conveyorTile, Entity entity, EnumFacing facing)
		{
			ConveyorDirection conveyorDirection = getConveyorDirection();
			BlockPos pos = conveyorTile.func_174877_v();

			double vBase = 1.15;
			double vX = 0.1 * vBase * facing.func_82601_c();
			double vY = entity.field_70181_x;
			double vZ = 0.1 * vBase * facing.func_82599_e();

			if(conveyorDirection == ConveyorDirection.UP)
				vY = 0.17D * vBase;
			else if(conveyorDirection == ConveyorDirection.DOWN)
				vY = -0.07000000000000001D * vBase;

			if(conveyorDirection != ConveyorDirection.HORIZONTAL)
				entity.field_70122_E = false;

			if(facing == EnumFacing.WEST || facing == EnumFacing.EAST)
			{
				if(entity.field_70161_v > pos.func_177952_p() + 0.55D)
					vZ = -0.1D * vBase;
				else if(entity.field_70161_v < pos.func_177952_p() + 0.45D)
					vZ = 0.1D * vBase;
			} else if(facing == EnumFacing.NORTH || facing == EnumFacing.SOUTH)
			{
				if(entity.field_70165_t > pos.func_177958_n() + 0.55D)
					vX = -0.1D * vBase;
				else if(entity.field_70165_t < pos.func_177958_n() + 0.45D)
					vX = 0.1D * vBase;
			}

			return new Vec3d(vX, vY, vZ);
		}

		default void onEntityCollision(TileEntity tile, Entity entity, EnumFacing facing)
		{
			if(!isActive(tile))
				return;
			BlockPos pos = tile.func_174877_v();
			ConveyorDirection conveyorDirection = getConveyorDirection();
			float heightLimit = conveyorDirection == ConveyorDirection.HORIZONTAL ? .25f : 1f;
			if(entity != null && !entity.field_70128_L && !(entity instanceof EntityPlayer && entity.func_70093_af()) && entity.field_70163_u - pos.func_177956_o() >= 0 && entity.field_70163_u - pos.func_177956_o() < heightLimit)
			{
				Vec3d vec = this.getDirection(tile, entity, facing);
				if(entity.field_70143_R < 3)
					entity.field_70143_R = 0;
				entity.field_70159_w = vec.field_72450_a;
				entity.field_70181_x = vec.field_72448_b;
				entity.field_70179_y = vec.field_72449_c;
				double distX = Math.abs(pos.func_177972_a(facing).func_177958_n() + .5 - entity.field_70165_t);
				double distZ = Math.abs(pos.func_177972_a(facing).func_177952_p() + .5 - entity.field_70161_v);
				double treshold = .9;
				boolean contact = facing.func_176740_k() == Axis.Z ? distZ < treshold : distX < treshold;
				if(contact && conveyorDirection == ConveyorDirection.UP && !tile.func_145831_w().func_180495_p(pos.func_177972_a(facing).func_177984_a()).func_185913_b())
				{
					double move = .4;
					entity.func_70107_b(entity.field_70165_t + move * facing.func_82601_c(), entity.field_70163_u + 1 * move, entity.field_70161_v + move * facing.func_82599_e());
				}
				if(!contact)
					ConveyorHandler.applyMagnetSupression(entity, (IConveyorTile) tile);
				else
				{
					BlockPos nextPos = tile.func_174877_v().func_177972_a(facing);
					if(!(Utils.getExistingTileEntity(tile.func_145831_w(), nextPos) instanceof IConveyorTile))
						ConveyorHandler.revertMagnetSupression(entity, (IConveyorTile) tile);
				}

				if(entity instanceof EntityItem)
				{
					((EntityItem) entity).func_174873_u();
					handleInsertion(tile, (EntityItem) entity, facing, conveyorDirection, distX, distZ);
				}
			}
		}

		/**
		 * Called when an item is inserted into the conveyor and deployed as an entity
		 */
		default void onItemDeployed(TileEntity tile, EntityItem entity, EnumFacing facing){	}

		default void handleInsertion(TileEntity tile, EntityItem entity, EnumFacing facing, ConveyorDirection conDir, double distX, double distZ)
		{
			BlockPos invPos = tile.func_174877_v().func_177972_a(facing).func_177982_a(0, (conDir == ConveyorDirection.UP ? 1 : conDir == ConveyorDirection.DOWN ? -1 : 0), 0);
			World world = tile.func_145831_w();
			TileEntity inventoryTile = Utils.getExistingTileEntity(world, invPos);
			boolean contact = facing.func_176740_k() == Axis.Z ? distZ < .7 : distX < .7;
			if (!tile.func_145831_w().field_72995_K)
			{
				if (contact && inventoryTile != null && !(inventoryTile instanceof IConveyorTile))
				{
					ItemStack stack = entity.func_92059_d();
					if (!stack.func_190926_b())
					{
						ItemStack ret = ApiUtils.insertStackIntoInventory(inventoryTile, stack, facing.func_176734_d());
						if (ret.func_190926_b())
							entity.func_70106_y();
						else if (ret.func_190916_E() < stack.func_190916_E())
							entity.func_92058_a(ret);
					}
				}
			}

		}

		AxisAlignedBB conveyorBounds = new AxisAlignedBB(0, 0, 0, 1, .125f, 1);
		AxisAlignedBB highConveyorBounds = new AxisAlignedBB(0, 0, 0, 1, 1.125f, 1);

		default List<AxisAlignedBB> getSelectionBoxes(TileEntity tile, EnumFacing facing)
		{
			return getConveyorDirection() == ConveyorDirection.HORIZONTAL ? Lists.newArrayList(conveyorBounds) : Lists.newArrayList(highConveyorBounds);
		}

		default List<AxisAlignedBB> getColisionBoxes(TileEntity tile, EnumFacing facing)
		{
			return Lists.newArrayList(conveyorBounds);
		}

		NBTTagCompound writeConveyorNBT();

		void readConveyorNBT(NBTTagCompound nbt);

		@SideOnly(Side.CLIENT)
		default Matrix4f modifyBaseRotationMatrix(Matrix4f matrix, @Nullable TileEntity tile, EnumFacing facing)
		{
			return matrix;
		}

		@SideOnly(Side.CLIENT)
		ResourceLocation getActiveTexture();

		@SideOnly(Side.CLIENT)
		ResourceLocation getInactiveTexture();

		@SideOnly(Side.CLIENT)
		default ResourceLocation getColouredStripesTexture(){ return textureConveyorColour; }

		@SideOnly(Side.CLIENT)
		default List<BakedQuad> modifyQuads(List<BakedQuad> baseModel, @Nullable TileEntity tile, EnumFacing facing)
		{
			return baseModel;
		}
	}

	public enum ConveyorDirection
	{
		HORIZONTAL,
		UP,
		DOWN
	}

	/**
	 * An interface to prevent conveyors from rendering a wall in the direction of this tile
	 */
	public interface IConveyorAttachable
	{
		EnumFacing getFacing();

		/**
		 * @return a rough indication of where this block will output things. Will determine if attached conveyors render a wall in the opposite direction
		 */
		EnumFacing[] sigOutputDirections();
	}
	/**
	 * This interface solely exists to mark a tile as conveyor, and have it ignored for insertion
	 */
	public interface IConveyorTile extends IConveyorAttachable
	{
		IConveyorBelt getConveyorSubtype();

		void setConveyorSubtype(IConveyorBelt conveyor);

		@Override
		default EnumFacing[] sigOutputDirections()
		{
			IConveyorBelt subtype = getConveyorSubtype();
			if(subtype!=null)
				return subtype.sigTransportDirections((TileEntity)this, this.getFacing());
			return new EnumFacing[0];
		}
	}
}
