/*
 * BluSunrize
 * Copyright (c) 2017
 *
 * This code is licensed under "Blu's License of Common Sense"
 * Details can be found in the license file in the root folder of this project
 */

package blusunrize.immersiveengineering.api.tool;

import blusunrize.immersiveengineering.api.ApiUtils;
import blusunrize.immersiveengineering.api.ComparableItemStack;
import blusunrize.immersiveengineering.api.crafting.IngredientStack;
import com.google.common.collect.ImmutableList;
import net.minecraft.block.Block;
import net.minecraft.block.BlockChorusPlant;
import net.minecraft.block.BlockCrops;
import net.minecraft.block.BlockStem;
import net.minecraft.block.properties.IProperty;
import net.minecraft.block.properties.PropertyInteger;
import net.minecraft.block.state.IBlockState;
import net.minecraft.client.renderer.BlockRendererDispatcher;
import net.minecraft.client.renderer.GlStateManager;
import net.minecraft.client.renderer.block.model.IBakedModel;
import net.minecraft.init.Blocks;
import net.minecraft.init.Items;
import net.minecraft.item.ItemStack;
import net.minecraft.tileentity.TileEntity;
import net.minecraft.util.EnumFacing;
import net.minecraft.util.ResourceLocation;
import net.minecraftforge.fluids.FluidRegistry;
import net.minecraftforge.fluids.FluidStack;
import net.minecraftforge.fml.relauncher.Side;
import net.minecraftforge.fml.relauncher.SideOnly;
import net.minecraftforge.oredict.OreDictionary;

import javax.annotation.Nullable;
import java.util.HashMap;
import java.util.HashSet;

/**
 * @author BluSunrize - 09.03.2017
 *         <p>
 *         A handler for plantgrowth within the belljars/cloches
 */
public class BelljarHandler
{
	private static HashSet<IPlantHandler> plantHandlers = new HashSet<>();
	private static HashMap<ComparableItemStack, ResourceLocation> soilTextureMap = new HashMap<>();
	private static HashSet<FluidFertilizerHandler> fluidFertilizers = new HashSet<>();
	private static HashSet<ItemFertilizerHandler> itemFertilizers = new HashSet<>();

	public static float solidFertilizerModifier = 1;
	public static float fluidFertilizerModifier = 1;

	public static void registerHandler(IPlantHandler handler)
	{
		plantHandlers.add(handler);
	}
	public static IPlantHandler getHandler(ItemStack seed)
	{
		if(seed.func_190926_b())
			return null;
		for(IPlantHandler handler : plantHandlers)
			if(handler.isValid(seed))
				return handler;
		return null;
	}

	public static void registerFluidFertilizer(FluidFertilizerHandler handler)
	{
		fluidFertilizers.add(handler);
	}
	public static FluidFertilizerHandler getFluidFertilizerHandler(FluidStack fluid)
	{
		if(fluid==null)
			return null;
		for(FluidFertilizerHandler handler : fluidFertilizers)
			if(handler.isValid(fluid))
				return handler;
		return null;
	}


	public static void registerItemFertilizer(ItemFertilizerHandler handler)
	{
		itemFertilizers.add(handler);
	}
	public static ItemFertilizerHandler getItemFertilizerHandler(ItemStack itemStack)
	{
		if(itemStack.func_190926_b())
			return null;
		for(ItemFertilizerHandler handler : itemFertilizers)
			if(handler.isValid(itemStack))
				return handler;
		return null;
	}
	public static void registerBasicItemFertilizer(final ItemStack stack, final float growthMultiplier)
	{
		registerItemFertilizer(new ItemFertilizerHandler()
		{
			@Override
			public boolean isValid(@Nullable ItemStack fertilizer)
			{
				return OreDictionary.itemMatches(stack, fertilizer, false);
			}
			@Override
			public float getGrowthMultiplier(ItemStack fertilizer, ItemStack seed, ItemStack soil, TileEntity tile)
			{
				return solidFertilizerModifier*growthMultiplier;
			}
		});
	}

	public static ResourceLocation getSoilTexture(ItemStack soil)
	{
		return soilTextureMap.get(new ComparableItemStack(soil,false));
	}

	public interface IPlantHandler extends IPlantRenderer
	{
		boolean isCorrectSoil(ItemStack seed, ItemStack soil);
		float getGrowthStep(ItemStack seed, ItemStack soil, float growth, TileEntity tile, float fertilizer, boolean render);
		ItemStack[] getOutput(ItemStack seed, ItemStack soil, TileEntity tile);
		default float resetGrowth(ItemStack seed, ItemStack soil, float growth, TileEntity tile, boolean render)
		{
			return 0;
		}
		default ResourceLocation getSoilTexture(ItemStack seed, ItemStack soil, TileEntity tile)
		{
			return null;
		}
	}
	public interface IPlantRenderer
	{
		boolean isValid(ItemStack seed);
		@SideOnly(Side.CLIENT)
		IBlockState[] getRenderedPlant(ItemStack seed, ItemStack soil, float growth, TileEntity tile);
		@SideOnly(Side.CLIENT)
		default boolean overrideRender(ItemStack seed, ItemStack soil, float growth, TileEntity tile, BlockRendererDispatcher blockRenderer)
		{
			return false;
		}
		@SideOnly(Side.CLIENT)
		default float getRenderSize(ItemStack seed, ItemStack soil, float growth, TileEntity tile)
		{
			return .875f;
		}
	}

	public interface FluidFertilizerHandler
	{
		boolean isValid(@Nullable FluidStack fertilizer);
		float getGrowthMultiplier(FluidStack fertilizer, ItemStack seed, ItemStack soil, TileEntity tile);
	}
	public interface ItemFertilizerHandler
	{
		boolean isValid(ItemStack fertilizer);
		float getGrowthMultiplier(ItemStack fertilizer, ItemStack seed, ItemStack soil, TileEntity tile);
	}

	private static HashMap<ComparableItemStack, IngredientStack> seedSoilMap = new HashMap<>();
	private static HashMap<ComparableItemStack, ItemStack[]> seedOutputMap = new HashMap<>();
	private static HashMap<ComparableItemStack, IBlockState[]> seedRenderMap = new HashMap<>();

	public abstract static class DefaultPlantHandler implements IPlantHandler
	{
		protected abstract HashSet<ComparableItemStack> getSeedSet();

		@Override
		public boolean isValid(ItemStack seed)
		{
			return seed!=null&&getSeedSet().contains(new ComparableItemStack(seed,false));
		}
		@Override
		public boolean isCorrectSoil(ItemStack seed, ItemStack soil)
		{
			IngredientStack reqSoil = seedSoilMap.get(new ComparableItemStack(seed,false));
			return reqSoil.matchesItemStack(soil);
		}
		@Override
		public float getGrowthStep(ItemStack seed, ItemStack soil, float growth, TileEntity tile, float fertilizer, boolean render)
		{
			return .003125f * fertilizer;
		}
		@Override
		public ItemStack[] getOutput(ItemStack seed, ItemStack soil, TileEntity tile)
		{
			return seedOutputMap.get(new ComparableItemStack(seed,false));
		}

		@Override
		@SideOnly(Side.CLIENT)
		public IBlockState[] getRenderedPlant(ItemStack seed, ItemStack soil, float growth, TileEntity tile)
		{
			IBlockState[] states = seedRenderMap.get(new ComparableItemStack(seed,false));
			if(states!=null)
				return states;
			return null;
		}

		public void register(ItemStack seed, ItemStack[] output, Object soil, IBlockState... render)
		{
			register(seed, output, ApiUtils.createIngredientStack(soil), render);
		}
		public void register(ItemStack seed, ItemStack[] output, IngredientStack soil, IBlockState... render)
		{
			ComparableItemStack comp = new ComparableItemStack(seed,false);
			getSeedSet().add(comp);
			seedSoilMap.put(comp, soil);
			seedOutputMap.put(comp, output);
			seedRenderMap.put(comp, render);
		}
	}

	public static DefaultPlantHandler cropHandler = new DefaultPlantHandler()
	{
		private HashSet<ComparableItemStack> validSeeds = new HashSet<>();
		@Override
		protected HashSet<ComparableItemStack> getSeedSet()
		{
			return validSeeds;
		}

		@Override
		@SideOnly(Side.CLIENT)
		public IBlockState[] getRenderedPlant(ItemStack seed, ItemStack soil, float growth, TileEntity tile)
		{
			IBlockState[] states = seedRenderMap.get(new ComparableItemStack(seed,false));
			if(states!=null)
			{
				IBlockState[] ret = new IBlockState[states.length];
				for(int i=0; i<states.length; i++)
					if(states[i]!=null)
						if(states[i].func_177230_c() instanceof BlockCrops)
						{
							int max = ((BlockCrops)states[i].func_177230_c()).func_185526_g();
							ret[i] = ((BlockCrops)states[i].func_177230_c()).func_185528_e(Math.min(max, Math.round(max*growth)));
						}
						else
						{
							for(IProperty prop : states[i].func_177227_a())
								if("age".equals(prop.func_177701_a())&&prop instanceof PropertyInteger)
								{
									int max = 0;
									for(Integer allowed : ((PropertyInteger)prop).func_177700_c())
										if(allowed!=null&&allowed > max)
											max = allowed;
									ret[i] = states[i].func_177226_a(prop, Math.min(max, Math.round(max*growth)));
								}
							if(ret[i]==null)
								ret[i] = states[i];
						}
				return ret;
			}
			return null;
		}
	};
	public static DefaultPlantHandler stemHandler = new DefaultPlantHandler()
	{
		private HashSet<ComparableItemStack> validSeeds = new HashSet<>();

		@Override
		protected HashSet<ComparableItemStack> getSeedSet()
		{
			return validSeeds;
		}

		@Override
		public float getGrowthStep(ItemStack seed, ItemStack soil, float growth, TileEntity tile, float fertilizer, boolean render)
		{
			return (growth<.5?.00625f:.0015625f) * fertilizer;
		}
		@Override
		public float resetGrowth(ItemStack seed, ItemStack soil, float growth, TileEntity tile, boolean render)
		{
			return .5f;
		}

		@Override
		@SideOnly(Side.CLIENT)
		public IBlockState[] getRenderedPlant(ItemStack seed, ItemStack soil, float growth, TileEntity tile)
		{
			return new IBlockState[0];
		}
		@Override
		@SideOnly(Side.CLIENT)
		public float getRenderSize(ItemStack seed, ItemStack soil, float growth, TileEntity tile)
		{
			return 1f;
		}
		@Override
		@SideOnly(Side.CLIENT)
		public boolean overrideRender(ItemStack seed, ItemStack soil, float growth, TileEntity tile, BlockRendererDispatcher blockRenderer)
		{
			ComparableItemStack comp = new ComparableItemStack(seed,false);
			IBlockState[] renderStates = seedRenderMap.get(comp);
			if(renderStates.length>0 && renderStates[0]!=null && renderStates[0].func_177230_c() instanceof BlockStem)
			{
				GlStateManager.func_179114_b(-90,0,1,0);
				BlockStem stem = (BlockStem)renderStates[0].func_177230_c();
				IBlockState state = stem.func_176223_P().func_177226_a(BlockStem.field_176484_a, (int)(growth >= .5?7: 2*growth*7));
				if(growth>=.5)
					state = state.func_177226_a(BlockStem.field_176483_b, EnumFacing.NORTH);
				IBakedModel model = blockRenderer.func_184389_a(state);
				GlStateManager.func_179109_b(.25f,.0625f,0);
				GlStateManager.func_179094_E();
				blockRenderer.func_175019_b().func_178266_a(model, state, 1, true);
				GlStateManager.func_179121_F();
				if(growth>=.5)
				{
					ItemStack[] fruit = seedOutputMap.get(new ComparableItemStack(seed,false));
					if(fruit!=null&&fruit.length>0&&!fruit[0].func_190926_b())
					{
						Block fruitBlock = Block.func_149634_a(fruit[0].func_77973_b());
						if(fruitBlock!=null)
						{
							state = fruitBlock.func_176223_P();
							model = blockRenderer.func_184389_a(state);
							GlStateManager.func_179094_E();
							float scale = (growth-.5f)*.5f;
							GlStateManager.func_179137_b(-scale/2, .5-scale, -.5+scale/2);
							GlStateManager.func_179152_a(scale, scale, scale);
							blockRenderer.func_175019_b().func_178266_a(model, state, 1, true);
							GlStateManager.func_179121_F();
						}
					}
				}
			}
			return true;
		}
	};
	public static DefaultPlantHandler stackingHandler = new DefaultPlantHandler()
	{
		private HashSet<ComparableItemStack> validSeeds = new HashSet<>();
		@Override
		protected HashSet<ComparableItemStack> getSeedSet()
		{
			return validSeeds;
		}

		@Override
		@SideOnly(Side.CLIENT)
		public IBlockState[] getRenderedPlant(ItemStack seed, ItemStack soil, float growth, TileEntity tile)
		{
			IBlockState[] states = seedRenderMap.get(new ComparableItemStack(seed,false));
			if(states!=null)
				return states;
			return null;
		}
		@Override
		@SideOnly(Side.CLIENT)
		public float getRenderSize(ItemStack seed, ItemStack soil, float growth, TileEntity tile)
		{
			IBlockState[] states = seedRenderMap.get(new ComparableItemStack(seed,false));
			if(states!=null && states.length>2)
				return .6875f-(states.length)*.0625f;
			return .6875f;
		}
		@Override
		@SideOnly(Side.CLIENT)
		public boolean overrideRender(ItemStack seed, ItemStack soil, float growth, TileEntity tile, BlockRendererDispatcher blockRenderer)
		{
			IBlockState[] states = seedRenderMap.get(new ComparableItemStack(seed,false));
			if(states!=null)
				GlStateManager.func_179109_b(0, (-1+growth)*(states.length-1), 0);
			return false;
		}
	};

	public static void init()
	{
		soilTextureMap.put(new ComparableItemStack(new ItemStack(Blocks.field_150346_d),false), new ResourceLocation("minecraft:blocks/farmland_wet"));
		registerHandler(cropHandler);
		registerHandler(stemHandler);
		registerHandler(stackingHandler);

		cropHandler.register(new ItemStack(Items.field_151014_N), new ItemStack[]{new ItemStack(Items.field_151015_O, 2),new ItemStack(Items.field_151014_N,1)}, new ItemStack(Blocks.field_150346_d), Blocks.field_150464_aj.func_176223_P());
		cropHandler.register(new ItemStack(Items.field_151174_bG), new ItemStack[]{new ItemStack(Items.field_151174_bG,2)}, new ItemStack(Blocks.field_150346_d), Blocks.field_150469_bN.func_176223_P());
		cropHandler.register(new ItemStack(Items.field_151172_bF), new ItemStack[]{new ItemStack(Items.field_151172_bF,2)}, new ItemStack(Blocks.field_150346_d), Blocks.field_150459_bM.func_176223_P());
		cropHandler.register(new ItemStack(Items.field_185163_cU), new ItemStack[]{new ItemStack(Items.field_185164_cV,2),new ItemStack(Items.field_185163_cU,1)}, new ItemStack(Blocks.field_150346_d), Blocks.field_185773_cZ.func_176223_P());
		cropHandler.register(new ItemStack(Items.field_151075_bm), new ItemStack[]{new ItemStack(Items.field_151075_bm,2)}, new ItemStack(Blocks.field_150425_aM), Blocks.field_150388_bm.func_176223_P());

		stemHandler.register(new ItemStack(Items.field_151080_bb), new ItemStack[]{new ItemStack(Blocks.field_150423_aK)}, new ItemStack(Blocks.field_150346_d), Blocks.field_150393_bb.func_176223_P());
		stemHandler.register(new ItemStack(Items.field_151081_bc), new ItemStack[]{new ItemStack(Blocks.field_150440_ba)}, new ItemStack(Blocks.field_150346_d), Blocks.field_150394_bc.func_176223_P());

		stackingHandler.register(new ItemStack(Items.field_151120_aE), new ItemStack[]{new ItemStack(Items.field_151120_aE,2)}, "sand", Blocks.field_150436_aH.func_176223_P(),Blocks.field_150436_aH.func_176223_P());
		stackingHandler.register(new ItemStack(Blocks.field_150434_aF), new ItemStack[]{new ItemStack(Blocks.field_150434_aF,2)}, "sand", Blocks.field_150434_aF.func_176223_P(),Blocks.field_150434_aF.func_176223_P());
		stackingHandler.register(new ItemStack(Blocks.field_185766_cS), new ItemStack[]{new ItemStack(Items.field_185161_cS,1)}, new ItemStack(Blocks.field_150377_bs), Blocks.field_185765_cR.func_176223_P().func_177226_a(BlockChorusPlant.field_185614_f,true).func_177226_a(BlockChorusPlant.field_185613_e,true),Blocks.field_185765_cR.func_176223_P().func_177226_a(BlockChorusPlant.field_185614_f,true).func_177226_a(BlockChorusPlant.field_185613_e,true),Blocks.field_185766_cS.func_176223_P());

		IngredientStack shroomSoil = new IngredientStack(ImmutableList.of(new ItemStack(Blocks.field_150391_bh), new ItemStack(Blocks.field_150346_d,1,2)));
		cropHandler.register(new ItemStack(Blocks.field_150337_Q), new ItemStack[]{new ItemStack(Blocks.field_150337_Q,2)}, shroomSoil, Blocks.field_150337_Q.func_176223_P());
		cropHandler.register(new ItemStack(Blocks.field_150338_P), new ItemStack[]{new ItemStack(Blocks.field_150338_P,2)}, shroomSoil, Blocks.field_150338_P.func_176223_P());

		registerFluidFertilizer(new FluidFertilizerHandler()
		{
			@Override
			public boolean isValid(@Nullable FluidStack fertilizer)
			{
				return fertilizer!=null&&fertilizer.getFluid()==FluidRegistry.WATER;
			}
			@Override
			public float getGrowthMultiplier(FluidStack fertilizer, ItemStack seed, ItemStack soil, TileEntity tile)
			{
				return fluidFertilizerModifier;
			}
		});
		registerItemFertilizer(new ItemFertilizerHandler()
		{
			final ItemStack bonemeal = new ItemStack(Items.field_151100_aR,1,15);
			@Override
			public boolean isValid(ItemStack fertilizer)
			{
				return !fertilizer.func_190926_b()&&OreDictionary.itemMatches(bonemeal,fertilizer,true);
			}
			@Override
			public float getGrowthMultiplier(ItemStack fertilizer, ItemStack seed, ItemStack soil, TileEntity tile)
			{
				return solidFertilizerModifier*1.25f;
			}
		});
	}
}
