/*
 * BluSunrize
 * Copyright (c) 2017
 *
 * This code is licensed under "Blu's License of Common Sense"
 * Details can be found in the license file in the root folder of this project
 */

package blusunrize.immersiveengineering.api.energy.wires;

import blusunrize.immersiveengineering.ImmersiveEngineering;
import blusunrize.immersiveengineering.api.ApiUtils;
import blusunrize.immersiveengineering.api.DimensionBlockPos;
import blusunrize.immersiveengineering.api.TargetingInfo;
import blusunrize.immersiveengineering.common.Config.IEConfig;
import blusunrize.immersiveengineering.common.IESaveData;
import blusunrize.immersiveengineering.common.util.IEDamageSources;
import net.minecraft.block.state.IBlockState;
import net.minecraft.entity.Entity;
import net.minecraft.entity.EntityLivingBase;
import net.minecraft.entity.item.EntityItem;
import net.minecraft.entity.player.EntityPlayer;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.tileentity.TileEntity;
import net.minecraft.util.*;
import net.minecraft.util.math.*;
import net.minecraft.world.IWorldEventListener;
import net.minecraft.world.World;
import net.minecraftforge.fml.common.FMLCommonHandler;
import net.minecraftforge.fml.relauncher.Side;
import org.apache.commons.lang3.tuple.ImmutablePair;
import org.apache.commons.lang3.tuple.ImmutableTriple;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.commons.lang3.tuple.Triple;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.BiConsumer;

import static blusunrize.immersiveengineering.api.ApiUtils.*;
import static java.util.Collections.newSetFromMap;

public class ImmersiveNetHandler
{
	public static final ImmersiveNetHandler INSTANCE = new ImmersiveNetHandler();
	public final BlockPlaceListener LISTENER = new BlockPlaceListener();
	public Map<Integer, ConcurrentHashMap<BlockPos, Set<Connection>>> directConnections = new ConcurrentHashMap<>();
	public Map<BlockPos, Set<AbstractConnection>> indirectConnections = new ConcurrentHashMap<>();
	public Map<BlockPos, Set<AbstractConnection>> indirectConnectionsNoOut = new ConcurrentHashMap<>();
	public Map<Integer, HashMap<Connection, Integer>> transferPerTick = new HashMap<>();
	public Map<DimensionBlockPos, IICProxy> proxies = new ConcurrentHashMap<>();

	public IntHashMap<Map<BlockPos, BlockWireInfo>> blockWireMap = new IntHashMap<>();
	
	private ConcurrentHashMap<BlockPos, Set<Connection>> getMultimap(int dimension)
	{
		if (directConnections.get(dimension) == null)
		{
			ConcurrentHashMap<BlockPos, Set<Connection>> mm = new ConcurrentHashMap<BlockPos, Set<Connection>>();
			directConnections.put(dimension, mm);
		}
		return directConnections.get(dimension);
	}
	public HashMap<Connection, Integer> getTransferedRates(int dimension)
	{
		if (!transferPerTick.containsKey(dimension))
			transferPerTick.put(dimension, new HashMap<Connection,Integer>());
		return transferPerTick.get(dimension);
	}

	public void addConnection(World world, BlockPos node, BlockPos connection, int distance, WireType cableType)
	{
		addAndGetConnection(world, node, connection, distance, cableType);
	}
	public Connection addAndGetConnection(World world, BlockPos node, BlockPos connection, int distance, WireType cableType)
	{
		Connection conn = new Connection(node, connection, cableType, distance);
		addConnection(world.field_73011_w.getDimension(), node, conn);
		addConnection(world.field_73011_w.getDimension(), connection, new Connection(connection, node, cableType, distance));
		if(world.func_175667_e(node))
			world.func_175641_c(node, world.func_180495_p(node).func_177230_c(),-1,0);
		if(world.func_175667_e(connection))
			world.func_175641_c(connection, world.func_180495_p(connection).func_177230_c(),-1,0);
		return conn;
	}
	public void addConnection(World world, BlockPos node, Connection con)
	{
		addConnection(world.field_73011_w.getDimension(), node, con);
	}
	public void addConnection(int world, BlockPos node, Connection con)
	{
		if(!getMultimap(world).containsKey(node))
			getMultimap(world).put(node, newSetFromMap(new ConcurrentHashMap<>()));
		getMultimap(world).get(node).add(con);
		resetCachedIndirectConnections();
		IESaveData.setDirty(world);
	}

	public <T extends TileEntity & IImmersiveConnectable> void onTEValidated(T te)
	{
		Set<Connection> conns = getConnections(te.func_145831_w(), te.func_174877_v());
		if (conns != null)
			for (Connection con : conns)
				addBlockData(te.func_145831_w(), con);
		resetCachedIndirectConnections();
	}

	public void addBlockData(World world, Connection con)
	{
		int dimId = world.field_73011_w.getDimension();
		if (!blockWireMap.func_76037_b(dimId))
			blockWireMap.func_76038_a(dimId, new ConcurrentHashMap<>());
		Map<BlockPos, BlockWireInfo> mapForDim = blockWireMap.func_76041_a(dimId);
		if (mapForDim==null||!world.func_175667_e(con.end))
			return;
		raytraceAlongCatenary(con, world, (p) ->
		{
			if (!mapForDim.containsKey(p.getLeft()))
				mapForDim.put(p.getLeft(), new BlockWireInfo());
			if (mapForDim.get(p.getLeft()).in.stream().noneMatch((c)->c.getLeft().hasSameConnectors(con)))
				mapForDim.get(p.getLeft()).in.add(new ImmutableTriple<>(con, p.getMiddle(), p.getRight()));
			return false;
		}, (p) ->
		{
			if (!mapForDim.containsKey(p.getLeft()))
				mapForDim.put(p.getLeft(), new BlockWireInfo());
			if (mapForDim.get(p.getLeft()).near.stream().noneMatch((c)->c.getLeft().hasSameConnectors(con)))
				mapForDim.get(p.getLeft()).near.add(new ImmutableTriple<>(con, p.getMiddle(), p.getRight()));
		});
	}


	public void removeConnection(World world, Connection con)
	{
		if (con == null || world == null)
			return;
		Map<BlockPos, Set<Connection>> connsInDim = getMultimap(world.field_73011_w.getDimension());
		Set<Connection> reverseConns = connsInDim.get(con.end);
		Set<Connection> forwardConns = connsInDim.get(con.start);
		Optional<Connection> back = reverseConns.stream().filter(con::hasSameConnectors).findAny();
		reverseConns.removeIf(con::hasSameConnectors);
		forwardConns.removeIf(con::hasSameConnectors);
		Map<BlockPos, BlockWireInfo> mapForDim = blockWireMap.func_76041_a(world.field_73011_w.getDimension());
		BiConsumer<BlockPos, Map<BlockPos, BlockWireInfo>> handle = (p, map)-> {
			if (mapForDim!=null)
			{
				BlockWireInfo info = map.get(p);
				if (info!=null)
				{
					for (int i = 0;i<2;i++)
					{
						Set<Triple<Connection, Vec3d, Vec3d>> s = i==0?info.in:info.near;
							s.removeIf((t) -> t.getLeft().hasSameConnectors(con));
							if (s.isEmpty())
								map.remove(p);
					}
					if (info.near.isEmpty()&&info.in.isEmpty())
						map.remove(p);
				}
			}
		};
		raytraceAlongCatenary(con, world, (p)->{
			handle.accept(p.getLeft(), mapForDim);
			return false;
		}, (p)->handle.accept(p.getLeft(), mapForDim));

		IImmersiveConnectable iic = toIIC(con.end, world);
		if (iic != null)
		{
			iic.removeCable(con);
			back.ifPresent(iic::removeCable);
		}
		iic = toIIC(con.start, world);
		if (iic != null)
		{
			iic.removeCable(con);
			back.ifPresent(iic::removeCable);
		}

		if (world.func_175667_e(con.start))
			world.func_175641_c(con.start, world.func_180495_p(con.start).func_177230_c(), -1, 0);
		if (world.func_175667_e(con.end))
			world.func_175641_c(con.end, world.func_180495_p(con.end).func_177230_c(), -1, 0);

		resetCachedIndirectConnections();
		IESaveData.setDirty(world.field_73011_w.getDimension());
	}
	public Set<Integer> getRelevantDimensions()
	{
		return directConnections.keySet();
	}
	public Collection<Connection> getAllConnections(int dimensionId)
	{
		Set<Connection> ret = newSetFromMap(new ConcurrentHashMap<Connection, Boolean>());
		for (Set<Connection> conlist : getMultimap(dimensionId).values())
			ret.addAll(conlist);
		return ret;
	}
	public Collection<Connection> getAllConnections(World world)
	{
		return getAllConnections(world.field_73011_w.getDimension());
	}
	@Nullable
	public synchronized Set<Connection> getConnections(World world, BlockPos node)
	{
		if(world!=null && world.field_73011_w!=null)
		{
			return getConnections(world.field_73011_w.getDimension(), node);
		}
		return null;
	}
	@Nullable
	public synchronized Set<Connection> getConnections(int world, BlockPos node)
	{
		ConcurrentHashMap<BlockPos, Set<Connection>> map = getMultimap(world);
		return map.get(node);
	}
	public void clearAllConnections(World world)
	{
		clearAllConnections(world.field_73011_w.getDimension());
	}
	public void clearAllConnections(int world)
	{
		getMultimap(world).clear();
		blockWireMap.func_76049_d(world);
	}
	public void clearConnectionsOriginatingFrom(BlockPos node, World world)
	{
		if(getMultimap(world.field_73011_w.getDimension()).containsKey(node))
			getMultimap(world.field_73011_w.getDimension()).get(node).clear();
		resetCachedIndirectConnections();
	}

	public void resetCachedIndirectConnections()
	{
		if(FMLCommonHandler.instance().getEffectiveSide() == Side.SERVER)
		{
			indirectConnections.clear();
			indirectConnectionsNoOut.clear();
		}
		else
			ImmersiveEngineering.proxy.clearConnectionModelCache();
	}

	public void removeConnectionAndDrop(Connection conn, World world, @Nullable BlockPos dropPos)
	{
		removeConnection(world, conn);
		if (dropPos != null)
		{
			double dx = dropPos.func_177958_n() + .5;
			double dy = dropPos.func_177956_o() + .5;
			double dz = dropPos.func_177952_p() + .5;
			if (world.func_82736_K().func_82766_b("doTileDrops"))
				world.func_72838_d(new EntityItem(world, dx, dy, dz, conn.cableType.getWireCoil(conn)));
		}
	}

	/**
	 * Clears all connections to and from this node.
	 */
	public void clearAllConnectionsFor(BlockPos node, World world, boolean doDrops)
	{
		IImmersiveConnectable iic = toIIC(node, world);
		if (iic != null)
			iic.removeCable(null);

		if(getMultimap(world.field_73011_w.getDimension()).containsKey(node))
		{
			for (Connection con : getMultimap(world.field_73011_w.getDimension()).get(node))
			{
				removeConnection(world, con);
				double dx = node.func_177958_n() + .5 + Math.signum(con.start.func_177958_n() - con.end.func_177958_n());
				double dy = node.func_177956_o() + .5 + Math.signum(con.start.func_177956_o() - con.end.func_177956_o());
				double dz = node.func_177952_p() + .5 + Math.signum(con.start.func_177952_p() - con.end.func_177952_p());
				if (doDrops && world.func_82736_K().func_82766_b("doTileDrops"))
					world.func_72838_d(new EntityItem(world, dx, dy, dz, con.cableType.getWireCoil(con)));
			}
		}
		IESaveData.setDirty(world.field_73011_w.getDimension());
	}


	public void setProxy(DimensionBlockPos pos, IICProxy p)
	{
		if (p==null)
			proxies.remove(pos);
		else
			proxies.put(pos, p);
	}
	public void addProxy(IICProxy p)
	{
		if (p==null)
			return;
		setProxy(new DimensionBlockPos(p.getPos(), p.getDimension()), p);
	}

	/**
	 * Clears all connections to and from this node.
	 */
	public boolean clearAllConnectionsFor(BlockPos node, World world, @Nonnull TargetingInfo target)
	{
		IImmersiveConnectable iic = toIIC(node, world);
		WireType type = iic.getCableLimiter(target);
		if(type==null)
			return false;
		boolean ret = false;
		for(Connection con : getMultimap(world.field_73011_w.getDimension()).get(node))
		{
			if (con.cableType == type)
			{
				removeConnection(world, con);
				double dx = node.func_177958_n() + .5 + Math.signum(con.start.func_177958_n() - con.end.func_177958_n());
				double dy = node.func_177956_o() + .5 + Math.signum(con.start.func_177956_o() - con.end.func_177956_o());
				double dz = node.func_177952_p() + .5 + Math.signum(con.start.func_177952_p() - con.end.func_177952_p());
				if (world.func_82736_K().func_82766_b("doTileDrops"))
					world.func_72838_d(new EntityItem(world, dx, dy, dz, con.cableType.getWireCoil(con)));
				ret = true;
			}
		}
		if(world.func_175667_e(node))
			world.func_175641_c(node, world.func_180495_p(node).func_177230_c(),-1,0);

		IESaveData.setDirty(world.field_73011_w.getDimension());
		resetCachedIndirectConnections();
		return ret;
	}

	/*
	public static List<IImmersiveConnectable> getValidEnergyOutputs(BlockPos node, World world)
	{
		List<IImmersiveConnectable> openList = new ArrayList<IImmersiveConnectable>();
		List<IImmersiveConnectable> closedList = new ArrayList<IImmersiveConnectable>();
		List<BlockPos> checked = new ArrayList<BlockPos>();
		HashMap<BlockPos,BlockPos> backtracker = new HashMap<BlockPos,BlockPos>();

		checked.add(node);
		for(Connection con : getConnections(world, node))
			if(toIIC(con.end, world)!=null)
			{
				openList.add(toIIC(con.end, world));
				backtracker.put(con.end, node);
			}

		IImmersiveConnectable next = null;
		final int closedListMax = 1200;

		while(closedList.size()<closedListMax && !openList.isEmpty())
		{
			next = openList.get(0);
			if(!checked.contains(toCC(next)))
			{
				if(next.isEnergyOutput())
				{
					BlockPos last = toCC(next);
					WireType averageType = null;
					int distance = 0;
					List<Connection> connectionParts = new ArrayList<Connection>();
					while(last!=null)
					{
						BlockPos prev = last;
						last = backtracker.get(last);
						if(last!=null)
						{
							for(Connection conB : getConnections(world, prev))
								if(conB.end.equals(toCC(last)))
								{
									connectionParts.add(conB);
									distance += conB.length;
									if(averageType==null || averageType.ordinal()>conB.cableType.ordinal())
										averageType = conB.cableType;
									break;
								}
						}
					}
					closedList.add(next);
				}

				for(Connection con : getConnections(world, toCC(next)))
					if(toIIC(con.end, world)!=null && !checked.contains(con.end) && !closedList.contains(toIIC(con.end, world)) && !openList.contains(toIIC(con.end, world)))
					{
						openList.add(toIIC(con.end, world));
						backtracker.put(con.end, toCC(next));
					}
				checked.add(toCC(next));
			}
			openList.remove(0);
		}

		return closedList;
	}
	 */
	
	public Set<AbstractConnection> getIndirectEnergyConnections(BlockPos node, World world)
	{
		return getIndirectEnergyConnections(node, world, false);
	}
	public Set<AbstractConnection> getIndirectEnergyConnections(BlockPos node, World world, boolean ignoreIsEnergyOutput)
	{
		if(!ignoreIsEnergyOutput&&indirectConnections.containsKey(node))
			return indirectConnections.get(node);
		else if(ignoreIsEnergyOutput&&indirectConnectionsNoOut.containsKey(node))
			return indirectConnectionsNoOut.get(node);

		PriorityQueue<Pair<IImmersiveConnectable, Float>> queue = new PriorityQueue<>(Comparator.comparingDouble(Pair::getRight));
		Set<AbstractConnection> closedList = newSetFromMap(new ConcurrentHashMap<AbstractConnection, Boolean>());
		List<BlockPos> checked = new ArrayList<>();
		HashMap<BlockPos,BlockPos> backtracker = new HashMap<>();

		checked.add(node);
		Set<Connection> conL = getConnections(world, node);
		if(conL!=null)
			for(Connection con : conL)
			{
				IImmersiveConnectable end = toIIC(con.end, world);
				if(end!=null)
				{
					queue.add(new ImmutablePair<>(end, con.getBaseLoss()));
					backtracker.put(con.end, node);
				}
			}

		IImmersiveConnectable next;
		final int closedListMax = 1200;

		while(closedList.size()<closedListMax && !queue.isEmpty())
		{
			Pair<IImmersiveConnectable, Float> pair = queue.poll();
			next = pair.getLeft();
			float loss = pair.getRight();
			BlockPos nextPos = toBlockPos(next);
			if(!checked.contains(nextPos)&&queue.stream().noneMatch((p)->p.getLeft().equals(nextPos)))
			{
				boolean isOutput = next.isEnergyOutput();
				if(ignoreIsEnergyOutput||isOutput)
				{
					BlockPos last = toBlockPos(next);
					WireType minimumType = null;
					int distance = 0;
					List<Connection> connectionParts = new ArrayList<>();
					while(last!=null)
					{
						BlockPos prev = last;
						last = backtracker.get(last);
						if(last!=null)
						{

							Set<Connection> conLB = getConnections(world, last);
							if(conLB!=null)
								for(Connection conB : conLB)
									if(conB.end.equals(prev))
									{
										connectionParts.add(0, conB);
										distance += conB.length;
										if(minimumType==null || conB.cableType.getTransferRate()<minimumType.getTransferRate())
											minimumType = conB.cableType;
										break;
									}
						}
					}
					closedList.add(new AbstractConnection(toBlockPos(node), toBlockPos(next), minimumType, distance, isOutput, connectionParts.toArray(new Connection[connectionParts.size()])));
				}

				Set<Connection> conLN = getConnections(world, toBlockPos(next));
				if(conLN!=null)
					for(Connection con : conLN)
						if(next.allowEnergyToPass(con))
						{
							IImmersiveConnectable end = toIIC(con.end, world);

							Optional<Pair<IImmersiveConnectable, Float>> existing =
									queue.stream().filter((p)->p.getLeft()==end).findAny();
							float newLoss = con.getBaseLoss()+loss;
							if(end!=null && !checked.contains(con.end) && existing.map(Pair::getRight).orElse(Float.MAX_VALUE)>newLoss)
							{
								existing.ifPresent(p1 -> queue.removeIf((p2) -> p1.getLeft() == p2.getLeft()));
								queue.add(new ImmutablePair<>(end, newLoss));
								backtracker.put(con.end, toBlockPos(next));
							}
						}
				checked.add(toBlockPos(next));
			}
		}
		if (FMLCommonHandler.instance().getEffectiveSide() == Side.SERVER)
		{
			if (ignoreIsEnergyOutput)
			{
				if (!indirectConnections.containsKey(node))
					indirectConnections.put(node, newSetFromMap(new ConcurrentHashMap<>()));
				indirectConnections.get(node).addAll(closedList);
			}
			else
			{
				if (!indirectConnectionsNoOut.containsKey(node))
					indirectConnectionsNoOut.put(node, newSetFromMap(new ConcurrentHashMap<>()));
				indirectConnectionsNoOut.get(node).addAll(closedList);
			}
		}
		return closedList;
	}
	//Called through ASM/coremod
	@SuppressWarnings("unused")
	public static void handleEntityCollision(BlockPos p, Entity e)
	{
		if (!e.field_70170_p.field_72995_K&&IEConfig.enableWireDamage&&e instanceof EntityLivingBase&&
				!e.func_180431_b(IEDamageSources.wireShock)&&
				!(e instanceof EntityPlayer&&((EntityPlayer) e).field_71075_bZ.field_75102_a))
		{
			Map<BlockPos, BlockWireInfo> mapForDim = INSTANCE.blockWireMap.func_76041_a(e.field_71093_bK);
			if (mapForDim!=null)
			{
				BlockWireInfo info = mapForDim.get(p);
				if (info!=null)
				{
					handleMapForDamage(info.in, (EntityLivingBase) e, p);
					handleMapForDamage(info.near, (EntityLivingBase) e, p);
				}
			}
		}
	}

	private static void handleMapForDamage(Set<Triple<Connection, Vec3d, Vec3d>> in, EntityLivingBase e, BlockPos here)
	{
		final double KNOCKBACK_PER_DAMAGE = 10;
		if (!in.isEmpty())
		{
			AxisAlignedBB eAabb = e.func_174813_aQ();
			for (Triple<Connection, Vec3d, Vec3d> conn : in)
				if (conn.getLeft().cableType.canCauseDamage())
				{
					double extra = conn.getLeft().cableType.getDamageRadius();
					AxisAlignedBB includingExtra = eAabb.func_186662_g(extra).func_72317_d(-here.func_177958_n(), -here.func_177956_o(), -here.func_177952_p());
					boolean endpointsInEntity = includingExtra.func_72318_a(conn.getMiddle())||
							includingExtra.func_72318_a(conn.getRight());
					RayTraceResult rayRes = endpointsInEntity?null:includingExtra.func_72327_a(conn.getMiddle(), conn.getRight());
					if (endpointsInEntity||(rayRes != null && rayRes.field_72313_a == RayTraceResult.Type.BLOCK))
					{
						IImmersiveConnectable iic = toIIC(conn.getLeft().start, e.field_70170_p);
						float damage = 0;
						if (iic != null)
							damage = iic.getDamageAmount(e, conn.getLeft());
						if (damage == 0)
						{
							iic = toIIC(conn.getLeft().end, e.field_70170_p);
							if (iic != null)
								damage = iic.getDamageAmount(e, conn.getLeft());
						}
						if (damage != 0 && e.func_70097_a(IEDamageSources.wireShock, damage))
						{
							Vec3d v = e.func_70040_Z();
							knockbackNoSource(e, damage/KNOCKBACK_PER_DAMAGE, v.field_72450_a, v.field_72449_c);
							iic.processDamage(e, damage, conn.getLeft());
						}
					}
				}
		}
	}

	public Connection getReverseConnection(int world, Connection ret)
	{
		return getConnections(world, ret.end).stream().filter(ret::hasSameConnectors).findAny().orElse(null);
	}

	public static class Connection implements Comparable<Connection>
	{
		public BlockPos start;
		public BlockPos end;
		public WireType cableType;
		public int length;
		public Vec3d[] catenaryVertices;
		public static final int vertices = 17;

		/**
		 * Used to calculate the catenary vertices:
		 * Y = a * cosh((( X-offsetX)/a)+offsetY
		 * (Y relative to start. X linear&horizontal, from 0 to horizontal length)
		 * set in getSubVertices
		 */
		public double catOffsetX;
		public double catOffsetY;
		public double catA;

		public Connection(BlockPos start, BlockPos end, WireType cableType, int length)
		{
			this.start=start;
			this.end=end;
			this.cableType=cableType;
			this.length=length;
		}

		public boolean hasSameConnectors(Connection con) {
			boolean n0 = start.equals(con.start)&&end.equals(con.end);
			boolean n1  =start.equals(con.end)&&end.equals(con.start);
			return n0||n1;
		}

		public Vec3d[] getSubVertices(Vec3d start, Vec3d end)
		{
			if(catenaryVertices==null)
				catenaryVertices = getConnectionCatenary(this, start, end);
			return catenaryVertices;
		}

		public Vec3d[] getSubVertices(World world)
		{
			return getSubVertices(getVecForIICAt(world, start, this),
					getVecForIICAt(world, end, this));
		}

		public Vec3d getVecAt(double pos, Vec3d vStart, Vec3d across, double lengthHor)
		{
			return vStart.func_72441_c(pos*across.field_72450_a, catA * Math.cosh((pos*lengthHor-catOffsetX)/catA)+catOffsetY,
					pos*across.field_72449_c);
		}

		public NBTTagCompound writeToNBT()
		{
			NBTTagCompound tag = new NBTTagCompound();
			if(start!=null)
				tag.func_74783_a("start", new int[]{start.func_177958_n(),start.func_177956_o(),start.func_177952_p()});
			if(end!=null)
				tag.func_74783_a("end", new int[]{end.func_177958_n(),end.func_177956_o(),end.func_177952_p()});
			tag.func_74778_a("cableType", cableType.getUniqueName());
			tag.func_74768_a("length", length);
			return tag;
		}
		public static Connection readFromNBT(NBTTagCompound tag)
		{
			if(tag==null)
				return null;
			int[] iStart = tag.func_74759_k("start");
			BlockPos start = new BlockPos(iStart[0],iStart[1],iStart[2]);

			int[] iEnd = tag.func_74759_k("end");
			BlockPos end = new BlockPos(iEnd[0],iEnd[1],iEnd[2]);

			WireType type = ApiUtils.getWireTypeFromNBT(tag, "cableType");

			if(start!=null && end!=null && type!=null)
				return new Connection(start,end, type, tag.func_74762_e("length"));
			return null;
		}

		@Override
		public int compareTo(@Nonnull Connection o)
		{
			if(this==o)
				return 0;
			int distComp = Integer.compare(length, o.length);
			int cableComp = -1*Integer.compare(cableType.getTransferRate(), o.cableType.getTransferRate());
			if(cableComp!=0)
				return cableComp;
			if (distComp!=0)
				return distComp;
			if (start.func_177958_n()!=o.start.func_177958_n())
				return start.func_177958_n()>o.start.func_177958_n()?1:-1;
				if (start.func_177956_o()!=o.start.func_177956_o())
					return start.func_177956_o()>o.start.func_177956_o()?1:-1;
					if (start.func_177952_p()!=o.start.func_177952_p())
						return start.func_177952_p()>o.start.func_177952_p()?1:-1;
						if (end.func_177958_n()!=o.end.func_177958_n())
							return end.func_177958_n()>o.end.func_177958_n()?1:-1;
							if (end.func_177956_o()!=o.end.func_177956_o())
								return end.func_177956_o()>o.end.func_177956_o()?1:-1;
								if (end.func_177952_p()!=o.end.func_177952_p())
									return end.func_177952_p()>o.end.func_177952_p()?1:-1;
									return 0;
		}
		@Override
		public boolean equals(Object obj)
		{
			if (!(obj instanceof Connection))
				return false;
			return compareTo((Connection)obj)==0;
		}

		@Override
		public int hashCode()
		{
			return Objects.hash(start, end, cableType);
		}

		public float getBaseLoss()
		{
			return getBaseLoss(0);
		}

		public float getBaseLoss(float mod)
		{
			float lengthRelative = this.length/(float)cableType.getMaxLength();
			return (float)(lengthRelative*cableType.getLossRatio()*(1+mod));
		}
	}

	public static class AbstractConnection extends Connection
	{
		public Connection[] subConnections;
		public boolean isEnergyOutput;
		public AbstractConnection(BlockPos start, BlockPos end, WireType cableType, int length, Connection... subConnections)
		{
			this(start, end, cableType, length, true, subConnections);
		}

		public AbstractConnection(BlockPos start, BlockPos end, WireType cableType, int length, boolean output, Connection... subConnections)
		{
			super(start,end,cableType,length);
			this.isEnergyOutput = output;
			this.subConnections=subConnections;
		}

		public float getPreciseLossRate(int energyInput, int connectorMaxInput)
		{
			float f = 0;
			for(Connection c : subConnections)
			{
				float mod = (((connectorMaxInput-energyInput)/(float)connectorMaxInput)/.25f)*.1f;
				f += c.getBaseLoss(mod);
			}
			return Math.min(f,1);
		}

		public float getAverageLossRate()
		{
			float f = 0;
			for(Connection c : subConnections)
			{
				f += c.getBaseLoss();
			}
			return Math.min(f,1);
		}

		@Override
		public boolean equals(Object o)
		{
			if (this == o) return true;
			if (o == null || getClass() != o.getClass()) return false;
			if (!super.equals(o)) return false;
			AbstractConnection that = (AbstractConnection) o;
			return Arrays.equals(subConnections, that.subConnections);
		}

		@Override
		public int hashCode()
		{
			int result = super.hashCode();
			result = 31 * result + Arrays.hashCode(subConnections);
			return result;
		}
	}

	public class BlockWireInfo
	{
		@Nonnull
		public final Set<Triple<Connection, Vec3d, Vec3d>> in = Collections.newSetFromMap(new ConcurrentHashMap<>());
		@Nonnull
		public final Set<Triple<Connection, Vec3d, Vec3d>> near = Collections.newSetFromMap(new ConcurrentHashMap<>());
	}

	public class BlockPlaceListener implements IWorldEventListener
	{
		@Override
		public void func_184376_a(@Nonnull World worldIn, @Nonnull BlockPos pos, @Nonnull IBlockState oldState, @Nonnull IBlockState newState, int flags)
		{
			Map<BlockPos, BlockWireInfo> worldMap = blockWireMap.func_76041_a(worldIn.field_73011_w.getDimension());
			if (worldMap != null && !worldIn.field_72995_K && (flags & 1) != 0 && newState.func_177230_c().func_176209_a(newState, false))
			{
				BlockWireInfo info = worldMap.get(pos);
				if (info != null)
				{
					Set<Triple<Connection, Vec3d, Vec3d>> conns = info.in;
					Set<Pair<Connection, BlockPos>> toBreak = new HashSet<>();
					for (Triple<Connection, Vec3d, Vec3d> conn : conns)
						if (!conn.getLeft().start.equals(pos) && !conn.getLeft().end.equals(pos))
						{
							BlockPos dropPos = pos;
							if (ApiUtils.preventsConnection(worldIn, pos, newState, conn.getMiddle(), conn.getRight()))
							{
								for (EnumFacing f : EnumFacing.field_82609_l)
									if (worldIn.func_175623_d(pos.func_177972_a(f)))
									{
										dropPos = dropPos.func_177972_a(f);
										break;
									}
								toBreak.add(new ImmutablePair<>(conn.getLeft(), dropPos));
							}
						}
					for (Pair<Connection, BlockPos> b : toBreak)
						removeConnectionAndDrop(b.getLeft(), worldIn, b.getRight());
				}
			}
		}

		@Override
		public void func_174959_b(@Nonnull BlockPos pos)
		{}

		@Override
		public void func_147585_a(int x1, int y1, int z1, int x2, int y2, int z2)
		{}

		@Override
		public void func_184375_a(EntityPlayer player, @Nonnull SoundEvent soundIn, @Nonnull SoundCategory category, double x, double y, double z, float volume, float pitch)
		{}

		@Override
		public void func_184377_a(@Nonnull SoundEvent soundIn, @Nonnull BlockPos pos)
		{}

		@Override
		public void func_180442_a(int particleID, boolean ignoreRange, double xCoord, double yCoord, double zCoord, double xSpeed, double ySpeed, double zSpeed, @Nonnull int... parameters)
		{}

		@Override
		public void func_190570_a(int id, boolean ignoreRange, boolean p_190570_3_, double x, double y, double z, double xSpeed, double ySpeed, double zSpeed, @Nonnull int... parameters)
		{}

		@Override
		public void func_72703_a(@Nonnull Entity entityIn)
		{}

		@Override
		public void func_72709_b(@Nonnull Entity entityIn)
		{}

		@Override
		public void func_180440_a(int soundID, @Nonnull BlockPos pos, int data)
		{}

		@Override
		public void func_180439_a(EntityPlayer player, int type, @Nonnull BlockPos blockPosIn, int data)
		{}

		@Override
		public void func_180441_b(int breakerId, @Nonnull BlockPos pos, int progress)
		{}
	}
}
