/*
 * BluSunrize
 * Copyright (c) 2017
 *
 * This code is licensed under "Blu's License of Common Sense"
 * Details can be found in the license file in the root folder of this project
 */

package blusunrize.immersiveengineering.api.crafting;

import blusunrize.immersiveengineering.api.ApiUtils;
import blusunrize.immersiveengineering.common.crafting.IngredientFluidStack;
import net.minecraft.item.ItemStack;
import net.minecraft.item.crafting.Ingredient;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.nbt.NBTTagList;
import net.minecraftforge.common.ForgeModContainer;
import net.minecraftforge.fluids.FluidRegistry;
import net.minecraftforge.fluids.FluidStack;
import net.minecraftforge.fluids.FluidUtil;
import net.minecraftforge.fluids.UniversalBucket;
import net.minecraftforge.oredict.OreDictionary;
import net.minecraftforge.oredict.OreIngredient;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

public class IngredientStack
{
	public ItemStack stack = ItemStack.field_190927_a;
	public List<ItemStack> stackList;
	public String oreName;
	public FluidStack fluid;
	public int inputSize = 1;
	public boolean useNBT;

	public IngredientStack(ItemStack stack)
	{
		this.stack = stack;
		this.inputSize = stack.func_190916_E();
	}
	public IngredientStack(String oreName, int inputSize)
	{
		this.oreName = oreName;
		this.inputSize = inputSize;
	}
	public IngredientStack(String oreName)
	{
		this(oreName, 1);
	}
	public IngredientStack(List<ItemStack> stackList, int inputSize)
	{
		this.stackList = stackList;
		this.inputSize = inputSize;
	}
	public IngredientStack(List<ItemStack> stackList)
	{
		this(stackList, 1);
	}
	public IngredientStack(FluidStack fluid)
	{
		this.fluid = fluid;
	}
	public IngredientStack(IngredientStack ingr)
	{
		this.stack = ingr.stack;
		this.stackList = ingr.stackList;
		this.oreName = ingr.oreName;
		this.fluid = ingr.fluid;
		this.inputSize = ingr.inputSize;
		this.useNBT = ingr.useNBT;
	}


	public IngredientStack setUseNBT(boolean useNBT)
	{
		this.useNBT = useNBT;
		return this;
	}

	public boolean matches(Object input)
	{
		if(input==null)
			return false;
		if(input instanceof IngredientStack)
			return this.equals(input) && this.inputSize <= ((IngredientStack)input).inputSize;
		if(input instanceof ItemStack)
		{
			return matchesItemStack((ItemStack)input);
		}
		else if(input instanceof ItemStack[])
		{
			for(ItemStack iStack : (ItemStack[])input)
				if(matchesItemStack(iStack))
					return true;
		}
		else if(input instanceof List)
		{
			for(Object io : (List)input)
				if(this.matches(io))
					return true;
		}
		else if(input instanceof String)
		{
			if(this.oreName!=null)
				return this.oreName.equals(input);
			return ApiUtils.compareToOreName(stack, (String)input);
		}
		return false;
	}

	public IngredientStack copyWithSize(int size)
	{
		IngredientStack is = new IngredientStack(this);
		is.inputSize = size;
		return is;
	}
	public IngredientStack copyWithMultipliedSize(double multiplier)
	{
		return copyWithSize((int)Math.floor(this.inputSize*multiplier));
	}

	public List<ItemStack> getStackList()
	{
		if(stackList!=null)
			return stackList;
		if(oreName!=null)
			return OreDictionary.getOres(oreName);
		if(fluid!=null&&ForgeModContainer.getInstance().universalBucket!=null)
			return Collections.singletonList(UniversalBucket.getFilledBucket(ForgeModContainer.getInstance().universalBucket, fluid.getFluid()));
		return Collections.singletonList(stack);
	}

	public List<ItemStack> getSizedStackList()
	{
		List<ItemStack> list;
		if(oreName!=null)
		{
			list = new ArrayList<ItemStack>();
			for(ItemStack stack : OreDictionary.getOres(oreName))
				list.add(ApiUtils.copyStackWithAmount(stack, inputSize));
		}
		else if(fluid!=null&&ForgeModContainer.getInstance().universalBucket!=null)
			list = Collections.singletonList(UniversalBucket.getFilledBucket(ForgeModContainer.getInstance().universalBucket, fluid.getFluid()));
		else if(stackList!=null)
			list = stackList;
		else
			list = Collections.singletonList(ApiUtils.copyStackWithAmount(stack, inputSize));
		return list;
	}

	public ItemStack getRandomizedExampleStack(long rand)
	{
		ItemStack ret = stack;
		if(ret.func_190926_b()&&stackList!=null&&stackList.size()>0)
			ret = stackList.get((int)(rand / 20) % stackList.size());
		if(ret.func_190926_b()&&oreName!=null)
		{
			List<ItemStack> ores = OreDictionary.getOres(oreName);
			if(ores!=null&&ores.size()>0)
				ret = ores.get((int)(rand / 20) % ores.size());
		}
		if(ret.func_190926_b()&&fluid!=null&&ForgeModContainer.getInstance().universalBucket!=null)
			ret = UniversalBucket.getFilledBucket(ForgeModContainer.getInstance().universalBucket, fluid.getFluid());
		return ret;
	}

	public ItemStack getExampleStack()
	{
		ItemStack ret = stack;
		if(ret.func_190926_b() && stackList != null && stackList.size() > 0)
			ret = stackList.get(0);
		if(ret.func_190926_b() && oreName != null)
		{
			List<ItemStack> ores = OreDictionary.getOres(oreName);
			if(ores != null && ores.size() > 0)
				ret = ores.get(0);
		}
		if(ret.func_190926_b() && fluid != null && ForgeModContainer.getInstance().universalBucket != null)
			ret = UniversalBucket.getFilledBucket(ForgeModContainer.getInstance().universalBucket, fluid.getFluid());
		return ret;
	}
	public Ingredient toRecipeIngredient()
	{
		Ingredient ret = stack!=null?Ingredient.func_193369_a(stack):null;
		if(ret==null&&stackList!=null&&stackList.size()>0)
			ret = ApiUtils.createIngredientFromList(stackList);
		if(ret==null&&oreName!=null)
			ret = new OreIngredient(oreName);
		if(ret==null&&fluid!=null&&ForgeModContainer.getInstance().universalBucket!=null)
			ret = new IngredientFluidStack(fluid);
		return ret;
	}

	public boolean matchesItemStack(ItemStack input)
	{
		if(input.func_190926_b())
			return false;
		if(this.fluid!=null)
		{
			FluidStack fs = FluidUtil.getFluidContained(input);
			if(fs!=null && fs.containsFluid(fluid))
				return true;
		}
		if(this.oreName!=null)
			return ApiUtils.compareToOreName(input, oreName) && this.inputSize <= input.func_190916_E();
		if(this.stackList!=null)
		{
			for(ItemStack iStack : this.stackList)
				if(OreDictionary.itemMatches(iStack, input, false) && this.inputSize <= input.func_190916_E())
					return true;
		}
		if(!OreDictionary.itemMatches(stack,input, false) || this.inputSize > input.func_190916_E())
			return false;
		if(this.useNBT)
		{
			if(this.stack.func_77942_o() != input.func_77942_o())
				return false;
			if(!this.stack.func_77942_o() && !input.func_77942_o())
				return true;
			return this.stack.func_77978_p().equals(input.func_77978_p());
		}
		return true;
	}

	public boolean matchesItemStackIgnoringSize(ItemStack input)
	{
		if(input.func_190926_b())
			return false;
		if(this.fluid!=null)
		{
			FluidStack fs = FluidUtil.getFluidContained(input);
			if(fs!=null && fs.containsFluid(fluid))
				return true;
		}
		if(this.oreName!=null)
			return ApiUtils.compareToOreName(input, oreName);
		if(this.stackList!=null)
		{
			for(ItemStack iStack : this.stackList)
				if(OreDictionary.itemMatches(iStack, input, false))
					return true;
		}
		if(!OreDictionary.itemMatches(stack,input, false))
			return false;
		if(this.useNBT)
		{
			if(this.stack.func_77942_o() != input.func_77942_o())
				return false;
			if(!this.stack.func_77942_o() && !input.func_77942_o())
				return true;
			return this.stack.func_77978_p().equals(input.func_77978_p());
		}
		return true;
	}

	@Override
	public boolean equals(Object object)
	{
		if(!(object instanceof IngredientStack))
			return false;
		if(this.fluid!=null && ((IngredientStack)object).fluid!=null)
			return this.fluid.equals(((IngredientStack)object).fluid);
		if(this.oreName!=null && ((IngredientStack)object).oreName!=null)
			return this.oreName.equals(((IngredientStack)object).oreName);
		if(this.stackList != null && ((IngredientStack) object).stackList != null)
		{
			for(ItemStack iStack : this.stackList)
				for(ItemStack iStack2 : ((IngredientStack) object).stackList)
					if(OreDictionary.itemMatches(iStack, iStack2, false))
						return true;
			return false;
		}
		if(!this.stack.func_190926_b() && !((IngredientStack)object).stack.func_190926_b())
		{
			ItemStack otherStack = ((IngredientStack)object).stack;
			if(!OreDictionary.itemMatches(stack,otherStack, false))
				return false;
			if(this.useNBT)
			{
				if(this.stack.func_77942_o() != otherStack.func_77942_o())
					return false;
				if(!this.stack.func_77942_o() && !otherStack.func_77942_o())
					return true;
				return this.stack.func_77978_p().equals(otherStack.func_77978_p());
			}
			return true;
		}
		return false;
	}

	public NBTTagCompound writeToNBT(NBTTagCompound nbt)
	{
		if(this.fluid!=null)
		{
			nbt.func_74778_a("fluid", FluidRegistry.getFluidName(fluid));
			nbt.func_74768_a("fluidAmount", fluid.amount);
			nbt.func_74768_a("nbtType", 3);
		}
		else if(this.oreName!=null)
		{
			nbt.func_74778_a("oreName", oreName);
			nbt.func_74768_a("nbtType", 2);
		}
		else if(this.stackList!=null)
		{
			NBTTagList list = new NBTTagList();
			for(ItemStack stack : stackList)
				if(!stack.func_190926_b())
					list.func_74742_a(stack.func_77955_b(new NBTTagCompound()));
			nbt.func_74782_a("stackList", list);
			nbt.func_74768_a("nbtType", 1);
		}
		else
		{
			nbt.func_74782_a("stack", stack.func_77955_b(new NBTTagCompound()));
			nbt.func_74768_a("nbtType", 0);
			nbt.func_74757_a("useNBT", useNBT);
		}
		nbt.func_74768_a("inputSize", inputSize);
		return nbt;
	}
	public static IngredientStack readFromNBT(NBTTagCompound nbt)
	{
		if(nbt.func_74764_b("nbtType"))
			switch(nbt.func_74762_e("nbtType"))
			{
				case 0:
					ItemStack stack = new ItemStack(nbt.func_74775_l("stack"));
					stack.func_190920_e(nbt.func_74762_e("inputSize"));
					IngredientStack ingr = new IngredientStack(stack);
					ingr.useNBT = nbt.func_74767_n("useNBT");
					return ingr;
				case 1:
					NBTTagList list = nbt.func_150295_c("stackList", 10);
					List<ItemStack> stackList = new ArrayList();
					for(int i=0; i<list.func_74745_c(); i++)
						stackList.add(new ItemStack(list.func_150305_b(i)));
					return new IngredientStack(stackList, nbt.func_74762_e("inputSize"));
				case 2:
					return new IngredientStack(nbt.func_74779_i("oreName"), nbt.func_74762_e("inputSize"));
				case 3:
					FluidStack fs = new FluidStack(FluidRegistry.getFluid(nbt.func_74779_i("fluid")),nbt.func_74762_e("fluidAmount"));
					return new IngredientStack(fs);
			}
		return null;
	}
}
